/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.date;


import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.ConverterFactory;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;

import java.time.*;
import java.time.temporal.Temporal;
import java.util.Date;

@SuppressWarnings({"unchecked", "rawtypes"})
public enum LongToJSR310ConverterFactory implements ConverterFactory<Long, Temporal> {
    INSTANCE;

    LongToJSR310ConverterFactory() {
    }

    public <T extends Temporal> Converter<Long, T> getConverter(Class<T> targetType) {
        return new LongToJSR310Converter(targetType);
    }

    private static class LongToJSR310Converter<T extends Temporal> implements Converter<Long, T> {
        private final Class<T> jsr310Type;

        public LongToJSR310Converter(Class<T> jsr310Type) {
            this.jsr310Type = jsr310Type;
        }

        public Class<Long> getSourceType() {
            return Long.class;
        }

        public Class<T> getTargetType() {
            return this.jsr310Type;
        }

        public T convert(Long source) {
            if (source == null) {
                return null;
            } else {
                Class<T> targetType = this.getTargetType();
                ZonedDateTime zondedDateTime = (new Date(source)).toInstant().atZone(ZoneId.systemDefault());
                if (targetType == LocalDateTime.class) {
                    return (T) zondedDateTime.toLocalDateTime();
                } else if (targetType == LocalDate.class) {
                    return (T) zondedDateTime.toLocalDate();
                } else if (targetType == LocalTime.class) {
                    return (T) zondedDateTime.toLocalTime();
                } else {
                    throw new ConverterNotFoundException(this.getSourceType(), targetType);
                }
            }
        }
    }
}
