/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.date;

import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;
import java.lang.reflect.ParameterizedType;
import java.sql.Time;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.temporal.Temporal;

public interface JSR310ToLongGenericConverter<S extends Temporal> extends Converter<S, Long> {
   default Class<S> getSourceType() {
      return (Class)((ParameterizedType)this.getClass().getGenericInterfaces()[0]).getActualTypeArguments()[0];
   }

   default Class<Long> getTargetType() {
      return Long.class;
   }

   default Long convert(S source) {
      if (source == null) {
         return null;
      } else {
         Class<S> sourceType = this.getSourceType();
         if (LocalDateTime.class.equals(sourceType)) {
            return ((LocalDateTime)source).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
         } else if (LocalDate.class.equals(sourceType)) {
            return ((LocalDate)source).atStartOfDay().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
         } else if (LocalTime.class.equals(sourceType)) {
            return Time.valueOf((LocalTime)source).getTime();
         } else {
            throw new ConverterNotFoundException(this.getSourceType(), Long.class);
         }
      }
   }

   public static enum LocalTimeToLongConverter implements JSR310ToLongGenericConverter<LocalTime> {
      INSTANCE;
   }

   public static enum LocalDateToLongConverter implements JSR310ToLongGenericConverter<LocalDate> {
      INSTANCE;
   }

   public static enum LocalDateTimeToLongConverter implements JSR310ToLongGenericConverter<LocalDateTime> {
      INSTANCE;
   }
}
