/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.date;

import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.exception.ConvertFailedException;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;
import org.springframework.util.ClassUtils;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.temporal.Temporal;
import java.util.Date;

public interface JSR310ToDateGenericConverter<S extends Temporal, T extends Date> extends Converter<S, T> {
    default T convert(S source) {
        if (source == null) {
            return null;
        } else {
            Class<T> targetType = this.getTargetType();
            Constructor<T> constructor = ClassUtils.getConstructorIfAvailable(targetType, Long.TYPE);
            Class<S> sourceType = this.getSourceType();
            if (constructor != null) {
                try {
                    long time;
                    if (LocalDateTime.class.equals(sourceType)) {
                        time = ((LocalDateTime)source).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
                    } else if (LocalDate.class.equals(sourceType)) {
                        time = ((LocalDate)source).atStartOfDay().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
                    } else {
                        if (!LocalTime.class.equals(sourceType)) {
                            throw new ConverterNotFoundException(this.getSourceType(), targetType);
                        }

                        time = Time.valueOf((LocalTime)source).getTime();
                    }

                    return constructor.newInstance(time);
                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | InstantiationException var7) {
                    throw new ConvertFailedException(this.getSourceType(), targetType, var7);
                }
            } else {
                throw new ConverterNotFoundException(this.getSourceType(), targetType);
            }
        }
    }

    public static enum LocalTimeToSqlDateConverter implements JSR310ToDateGenericConverter<LocalTime, java.sql.Date> {
        INSTANCE;

        private LocalTimeToSqlDateConverter() {
        }
    }

    public static enum LocalTimeToSqlTimestampConverter implements JSR310ToDateGenericConverter<LocalTime, Timestamp> {
        INSTANCE;

        private LocalTimeToSqlTimestampConverter() {
        }
    }

    public static enum LocalTimeToSqlTimeConverter implements JSR310ToDateGenericConverter<LocalTime, Time> {
        INSTANCE;

        private LocalTimeToSqlTimeConverter() {
        }
    }

    public static enum LocalTimeToDateConverter implements JSR310ToDateGenericConverter<LocalTime, Date> {
        INSTANCE;

        private LocalTimeToDateConverter() {
        }
    }

    public static enum LocalDateToSqlDateConverter implements JSR310ToDateGenericConverter<LocalDate, java.sql.Date> {
        INSTANCE;

        private LocalDateToSqlDateConverter() {
        }
    }

    public static enum LocalDateToSqlTimestampConverter implements JSR310ToDateGenericConverter<LocalDate, Timestamp> {
        INSTANCE;

        private LocalDateToSqlTimestampConverter() {
        }
    }

    public static enum LocalDateToSqlTimeConverter implements JSR310ToDateGenericConverter<LocalDate, Time> {
        INSTANCE;

        private LocalDateToSqlTimeConverter() {
        }
    }

    public static enum LocalDateToDateConverter implements JSR310ToDateGenericConverter<LocalDate, Date> {
        INSTANCE;

        private LocalDateToDateConverter() {
        }
    }

    public static enum LocalDateTimeToSqlDateConverter implements JSR310ToDateGenericConverter<LocalDateTime, java.sql.Date> {
        INSTANCE;

        private LocalDateTimeToSqlDateConverter() {
        }
    }

    public static enum LocalDateTimeToSqlTimestampConverter implements JSR310ToDateGenericConverter<LocalDateTime, Timestamp> {
        INSTANCE;

        private LocalDateTimeToSqlTimestampConverter() {
        }
    }

    public static enum LocalDateTimeToSqlTimeConverter implements JSR310ToDateGenericConverter<LocalDateTime, Time> {
        INSTANCE;

        private LocalDateTimeToSqlTimeConverter() {
        }
    }

    public static enum LocalDateTimeToDateConverter implements JSR310ToDateGenericConverter<LocalDateTime, Date> {
        INSTANCE;

        private LocalDateTimeToDateConverter() {
        }
    }
}
