/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.date;


import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.ConverterFactory;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;

import java.time.*;
import java.time.temporal.Temporal;
import java.util.Date;

@SuppressWarnings({"unused", "unchecked", "rawtypes"})
public enum DateToJSR310ConverterFactory implements ConverterFactory<Date, Temporal> {
    INSTANCE;

    DateToJSR310ConverterFactory() {
    }

    public <T extends Temporal> Converter<Date, T> getConverter(Class<T> targetType) {
        return new DateToJSR310Converter(targetType);
    }

    @SuppressWarnings("unchecked")
    private static class DateToJSR310Converter<T extends Temporal> implements Converter<Date, T> {
        private final Class<T> jsr310Type;

        public DateToJSR310Converter(Class<T> jsr310Type) {
            this.jsr310Type = jsr310Type;
        }

        public Class<Date> getSourceType() {
            return Date.class;
        }

        public Class<T> getTargetType() {
            return this.jsr310Type;
        }

        public T convert(Date source) {
            if (source == null) {
                return null;
            } else {
                Class<T> targetType = this.getTargetType();
                ZonedDateTime zondedDateTime = (java.sql.Date.class.equals(source.getClass()) ? new Date(source.getTime()) : source).toInstant().atZone(ZoneId.systemDefault());
                if (targetType == LocalDateTime.class) {
                    return (T) zondedDateTime.toLocalDateTime();
                } else if (targetType == LocalDate.class) {
                    return (T) zondedDateTime.toLocalDate();
                } else if (targetType == LocalTime.class) {
                    return (T) zondedDateTime.toLocalTime();
                } else {
                    throw new ConverterNotFoundException(this.getSourceType(), targetType);
                }
            }
        }
    }
}
