/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.context.concurrent;

import net.sinodawn.framework.cache.redis.RedisHelper;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.exception.ConcurrentException;
import net.sinodawn.framework.i18n.I18nHelper;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

public class SinoConcurrentLocker {
   public static final String LOCKER_ISOLATED_CACHE_NAME = "SINO_ISOLATED_CONCURRENT";
   public static final String LOCKER_BLOCK_CACHE_NAME = "SINO_BLOCK_CONCURRENT";
   public static final int DEFAULT_EXPIRE_SECONDS = 300;

   public static void isolated(String key) {
      String cacheValue = getCacheValue();
      if (!RedisHelper.putIfAbsent("SINO_ISOLATED_CONCURRENT", key, cacheValue, 300)) {
         if (!cacheValue.equals(RedisHelper.get("SINO_ISOLATED_CONCURRENT", key))) {
            throw new ConcurrentException(I18nHelper.getMessage("SINO.CONCURRENT.ISOLATED"));
         }
      } else {
         TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
            public void afterCompletion(int status) {
               SinoConcurrentLocker.releaseLock("SINO_ISOLATED_CONCURRENT", key);
            }
         });
      }
   }

   public static void isolatedRun(String key, Runnable runnable) {
      boolean lock = false;

      try {
         String cacheValue = getCacheValue();
         if (!RedisHelper.putIfAbsent("SINO_ISOLATED_CONCURRENT", key, cacheValue, 300) && !cacheValue.equals(RedisHelper.get("SINO_ISOLATED_CONCURRENT", key))) {
            return;
         }

         lock = true;
         runnable.run();
      } finally {
         if (lock) {
            releaseLock("SINO_ISOLATED_CONCURRENT", key);
         }

      }

   }

   public static void block(String key) {
      boolean block = true;
      String cacheValue = getCacheValue();

      while(!RedisHelper.putIfAbsent("SINO_BLOCK_CONCURRENT", key, cacheValue, 300)) {
         if (cacheValue.equals(RedisHelper.get("SINO_BLOCK_CONCURRENT", key))) {
            block = false;
            break;
         }

         try {
            Thread.sleep(50L);
         } catch (InterruptedException var4) {
         }
      }

      if (block) {
         TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
            public void afterCompletion(int status) {
               SinoConcurrentLocker.releaseLock("SINO_BLOCK_CONCURRENT", key);
            }
         });
      }

   }

   public static void blockRun(String key, int timeoutSeconds, Runnable runnable) {
      boolean block = true;

      try {
         if (timeoutSeconds <= 0) {
            timeoutSeconds = 300;
         }

         String cacheValue = getCacheValue();

         while(true) {
            if (!RedisHelper.putIfAbsent("SINO_BLOCK_CONCURRENT", key, cacheValue, timeoutSeconds)) {
               if (!cacheValue.equals(RedisHelper.get("SINO_BLOCK_CONCURRENT", key))) {
                  try {
                     Thread.sleep(50L);
                  } catch (InterruptedException var9) {
                  }
                  continue;
               }

               block = false;
            }

            runnable.run();
            return;
         }
      } finally {
         if (block) {
            releaseLock("SINO_BLOCK_CONCURRENT", key);
         }

      }
   }

   public static void blockRun(String key, Runnable runnable) {
      blockRun(key, 0, runnable);
   }

   public static void releaseLock(String name, String key) {
      try {
         RedisHelper.evict(name, key);
      } catch (Exception var3) {
      }

   }

   private static String getCacheValue() {
      return ApplicationContextHelper.getAppId() + Thread.currentThread().getId();
   }
}
