/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.context;

import net.sinodawn.framework.utils.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;

public class SinoAopContext {
   private static final ThreadLocal<Map<String, Object>> PROXY_CACHE = new ThreadLocal();
   private static final ThreadLocal<Object> CURRENT_PROXY = new ThreadLocal();

   public static <T> T currentProxy() {
      return (T) CURRENT_PROXY.get();
   }

   public static void setCurrentProxy(Object proxy) {
      if (proxy == null) {
         CURRENT_PROXY.remove();
      } else {
         CURRENT_PROXY.set(proxy);
      }

   }

   public static boolean initCache() {
      if (PROXY_CACHE.get() != null) {
         return false;
      } else {
         PROXY_CACHE.set(new HashMap());
         return true;
      }
   }

   public static void clearCache() {
      PROXY_CACHE.remove();
   }

   public static void removeCache(String key) {
      ((Map)PROXY_CACHE.get()).remove(key);
   }

   public static void putCache(String key, Object value) {
      ((Map)PROXY_CACHE.get()).put(key, value);
   }

   public static <T> T getCache(String key) {
      Map<String, Object> map = PROXY_CACHE.get();
      return map == null ? null : (T) map.get(key);
   }

   public static <T> T getCache(String key, Supplier<T> supplier) {
      T item = getCache(key);
      if (item == null) {
         item = supplier.get();
         putCache(key, item);
      }

      return item;
   }

   public static void clearCurrentCache() {
      Map<String, Object> map = (Map)PROXY_CACHE.get();
      if (map != null) {
         map.keySet().removeIf((k) -> {
            return StringUtils.startsWith(k, getCurrentCacheKeyPrefix());
         });
      }
   }

   public static void removeCurrentCache(String key) {
      ((Map)PROXY_CACHE.get()).remove(getCurrentCacheKeyPrefix() + key);
   }

   public static void putCurrentCache(String key, Object value) {
      ((Map)PROXY_CACHE.get()).put(getCurrentCacheKeyPrefix() + key, value);
   }

   public static <T> T getCurrentCache(String key) {
      Map<String, Object> map = PROXY_CACHE.get();
      return map == null ? null : (T) map.get(getCurrentCacheKeyPrefix() + key);
   }

   public static <T> T getCurrentCache(String key, Supplier<T> supplier) {
      T item = getCache(getCurrentCacheKeyPrefix() + key);
      if (item == null) {
         item = supplier.get();
         putCache(getCurrentCacheKeyPrefix() + key, item);
      }

      return item;
   }

   private static String getCurrentCacheKeyPrefix() {
      return currentProxy() + "$";
   }
}
