/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.boot;

import com.zaxxer.hikari.HikariDataSource;
import net.sinodawn.framework.database.dialect.Dialect;
import net.sinodawn.framework.database.dialect.MysqlDialect;
import org.apache.ibatis.logging.stdout.StdOutImpl;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.context.support.ApplicationObjectSupport;
import org.springframework.core.env.Environment;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.web.client.RestTemplate;

import javax.sql.DataSource;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@SuppressWarnings("unused")
@org.springframework.context.annotation.Configuration
@EnableTransactionManagement
@EnableAsync
@EnableScheduling
@EnableCaching
public class SinoApplicationConfiguration  extends ApplicationObjectSupport implements SchedulingConfigurer {

    @Autowired
    Environment env;

    @Autowired
    DataSource dataSource;

    @Bean
    @Primary
    public RestTemplate restTemplate(RestTemplateBuilder builder) {
        return new RestTemplate();
    }

    @Bean
    public ThreadPoolExecutor threadPoolExecutor() {
        return new ThreadPoolExecutor(4, 8, 1000, TimeUnit.MILLISECONDS, new ArrayBlockingQueue<>(5));
    }

    @Bean
    public DataSource dataSource() {
        HikariDataSource var = new HikariDataSource();
        var.setJdbcUrl(env.getProperty("spring.datasource.url"));
        var.setUsername(env.getProperty("spring.datasource.username"));
        var.setPassword(env.getProperty("spring.datasource.password"));
        return var;
    }

    @Bean
    public SqlSessionFactory sqlSessionFactory() {

        SqlSessionFactoryBean sqlSessionFactoryBean = new SqlSessionFactoryBean();
        sqlSessionFactoryBean.setDataSource(dataSource);
        Configuration configuration = new Configuration();
        configuration.setLogImpl(StdOutImpl.class);
        sqlSessionFactoryBean.setConfiguration(configuration);

        PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
        try {
            // TODO 根据properties自动判断mapper类型
            sqlSessionFactoryBean.setMapperLocations(
                    resolver.getResources("classpath*:/**/*mysql-mapper.xml")
            );
            return sqlSessionFactoryBean.getObject();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Bean
    RestTemplateBuilder restTemplateBuilder() { return new RestTemplateBuilder(); }

    @Bean
    RedisConnectionFactory redisConnectionFactory() { return new LettuceConnectionFactory(); }

    @Bean
    RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(redisConnectionFactory);
        redisTemplate.setKeySerializer(StringRedisSerializer.UTF_8);
        redisTemplate.setHashKeySerializer(StringRedisSerializer.UTF_8);
        return redisTemplate;
    }

    @Bean
    public Logger logger() {
        return LoggerFactory.getLogger("LOG");
    }

    @Bean
    public Dialect dialect() {
        // TODO 判断数据库类型
        return new MysqlDialect();
    }

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {

    }
}
