package zen.xml;

import java.io.File;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import zen.string.StringUtility;

public final class XmlWriter
{
	private XmlWriter()
	{
		super();
		
		//Empty constructor
	}
	
	public static void write(final XmlDocument xmlDocument, final String path) throws XmlException
	{
    	write(xmlDocument, new File(path));
	}
    
    public static void write(final XmlDocument xmlDocument, final File file) throws XmlException
	{
		final Document document = getDocument();
		final XmlNode node = xmlDocument.getRoot(); 
		final Element element = document.createElement(node.getName());
		
		setElement(document, element, node);
		document.appendChild(element);
		
		if (node.getChildren() != null)
		{
			setChildren(document, element, node);
		}
		
		transform(document, file);
	}
    
    private static void transform(final Document document, final File file) throws XmlException
    {
    	try
    	{
    		final Transformer transformer = getTransformer();
    		final DOMSource source = new DOMSource(document);
    		final StreamResult result = new StreamResult(file);
    		transformer.setOutputProperty(OutputKeys.INDENT, "yes");
    		transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
    		transformer.transform(source, result);	
    	}
    	catch (TransformerException exception)
    	{
    		throw new XmlException(exception);
    	}
    }
    
    private static void setElement(final Document document, final Element element, final XmlNode node) 
    {
    	if (!StringUtility.isEmpty(node.getValue()))
    	{
    		element.appendChild(document.createTextNode(node.getValue()));
    	}
    	
    	for (String name : node.getAttributeNames())
    	{
    		element.setAttribute(name, node.getAttribute(name));
    	}
    }

    private static void setChildren(final Document document, final Element parent, final XmlNode node) 
    {
    	for (XmlNode child : node.getChildren())
    	{
    		Element element = document.createElement(child.getName());
    		setElement(document, element, child);
    		parent.appendChild(element);
    		
    		if (child.getChildren() != null)
    		{
    			setChildren(document, element, child);
    		}
    	}
    }
    
    private static Transformer getTransformer() throws XmlException
    {
    	try
    	{
    		final TransformerFactory factory = TransformerFactory.newInstance();
    		return factory.newTransformer();	
    	}
    	catch (TransformerConfigurationException exception)
    	{
    		throw new XmlException(exception);
    	}
    }
	
	private static Document getDocument() throws XmlException
	{
		try
		{
			final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			final DocumentBuilder builder = factory.newDocumentBuilder();
		    return builder.newDocument();
		}
		catch (ParserConfigurationException exception)
		{
			throw new XmlException(exception);
		}
	}
}
