package zen.xml;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;

import org.apache.xerces.parsers.DOMParser;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import zen.file.FileException;
import zen.file.FileUtility;

public final class XmlParser 
{
	private XmlParser()
	{
		super();
		
		//Empty constructor
	}

    public static void load(final File file, final XmlDocument document) throws XmlException
    {
        load(getDocumentRoot(file), document);
    }

    public static void load(final InputStream stream, final XmlDocument document) throws XmlException
    {
        load(getDocumentRoot(stream), document);
    }

    public static void load(final String file, final XmlDocument document) throws XmlException
    {
    	load(getDocumentRoot(file), document);
    }
    
    public static void load(final StringBuffer input, final XmlDocument document) throws XmlException
    {
        load(getDocumentRoot(input), document);
    }
    
    private static void load(final Element root, final XmlDocument document) throws XmlException
    {
        document.setRoot(new XmlNode());
        parse(root, document.getRoot());
    }
	
    private static Element getDocumentRoot(final InputStream stream) throws XmlException
    {
        return getDocumentRoot(new InputSource(stream));
    }
    
    private static Element getDocumentRoot(final StringBuffer input) throws XmlException
    {
        return getDocumentRoot(new InputSource(new StringReader(input.toString())));
    }

    private static Element getDocumentRoot(final String file) throws XmlException
    {
        try
        {
            return getDocumentRoot(FileUtility.getStream(file));
        }
        catch (FileException exception)
        {
            throw new XmlException("Not able to get document root element for File [" + file + "]", exception);
        }
    }

    private static Element getDocumentRoot(final File file) throws XmlException
    {
        try
        {
            return getDocumentRoot(new InputSource(new FileReader(file)));
        }
        catch (FileNotFoundException exception)
        {
            throw new XmlException("Not able to get document root element for File [" + file.getPath() + "]", exception);
        }
    }
    
    private static Element getDocumentRoot(final InputSource source) throws XmlException
    {
        try
        {
            final DOMParser parser = new DOMParser();
            parser.parse(source);
            
            final Document document = parser.getDocument();
            return document.getDocumentElement();
        }
        catch (SAXException exception)
        {
            throw new XmlException("Unable to parse the input source with SAX", exception);
        }
        catch (IOException exception)
        {
            throw new XmlException("Unable to read the input source from the stream", exception);
        }
    }

    private static void parse(final Node node, final XmlNode xmlNode) 
    {
        if (node != null)
        {
            xmlNode.setName(node.getNodeName().trim());

            if (node.getFirstChild() != null)
            {
                xmlNode.setValue(node.getFirstChild().getNodeValue());
            }

            setAttributes(node, xmlNode);
            setChildren(node, xmlNode);
        }
    }
    
    private static void setAttributes(final Node node, final XmlNode xmlNode) 
    {
    	final NamedNodeMap map = node.getAttributes();

        for (int i = 0; map != null && i < map.getLength(); i++)
        {
        	final Node attribute = map.item(i);
            xmlNode.setAttribute(attribute.getNodeName().trim(), attribute.getNodeValue().trim());
        }
    }

    private static void setChildren(final Node node, final XmlNode xmlNode) 
    {
        final NodeList children = node.getChildNodes();

        for (int i = 0; children != null && i < children.getLength(); i++)
        {
            final Node child = children.item(i);

            if (child.getNodeName().trim().equals("#text"))
            {
                if (node.getNodeValue() != null)
                {
                    xmlNode.setValue(node.getNodeValue().trim());
                }
            }
            else
            {
            	final XmlNode xmlChild = new XmlNode();
                xmlNode.addChild(xmlChild);
                parse(child, xmlChild);
            }
        }
    }
}
