package zen.validation.implementations;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;

import zen.string.StringUtility;
import zen.validation.abstracts.AbstractValidation;

/**
 * Determines if the string value is a parseable date. Default Format:
 * MM/dd/yyyy<br/> By default, the "MM/dd/yyyy" format is used. You may specify
 * another format to use in the validation-mappings.xml file. Nest the following
 * tag within the &lt;rule&gt; tag:<br/>
 * <code>&lt;param name="format" value="<i>format</i>" /&gt;</code><br/>
 * Use formats that are valid for SimpleDateFormat.
 *
 * @author DCano
 */
public class Date extends AbstractValidation
{
	/**
	 * The format used to parse the date. If no format is defined in
	 * validation-mappings.xml, the default format is your locale-specific
	 * version of the two-digit month, two-digit day and four-digit year.<br/>
	 * Defined as nested tag within the &lt;rule&gt; in validation-mappings.xml
	 * as <code>&lt;param name="format" value="<i>format</i>" /&gt;</code>.
	 */
	public static final String	FORMAT	= "format";

	/**
	 * Locale-specific formatter for the base format of two-digit month,
	 * two-digit day and four-digit year
	 */
	private static final String	DEFAULT_FORMAT	= getDefaultFormat();

	/**
	 * Validates a date string to ensure it can be converted to a date. By
	 * default, the "MM/dd/yyyy" format is used. You may specify another format
	 * to use in the validation-mappings.xml file. Nest the following tag within
	 * the &lt;rule&gt; tag:<br/>
	 * <code>&lt;param name="format" value="<i>format</i>" /&gt;</code><br/>
	 * Use formats that are valid for SimpleDateFormat.
	 *
	 * @param value String
	 * @return boolean
	 */
	public boolean isValid(final String value)
	{
		String format = this.getParamValue(FORMAT);

		if (StringUtility.isEmpty(format))
		{
			format = DEFAULT_FORMAT;
		}

		final SimpleDateFormat sdf = new SimpleDateFormat(format, Locale.US);

		try
		{
			sdf.parse(value);
			return true;
		}
		catch (ParseException e)
		{
			return false;
		}
	}

	private static String getDefaultFormat()
	{
		// really wanted to make this locale-specific, but couldn't figure out how
		return "MM/dd/yyyy";
	}
}
