package zen.validation.abstracts;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import zen.logging.interfaces.ILogging;
import zen.string.StringUtility;
import zen.validation.interfaces.IValidation;

/**
 * This is the base validation that implements the IValidation contract and gives
 * those classes that inherit from it a simple Regular Expression interface.
 *
 * @author DCano
 *
 */
public abstract class AbstractValidation implements IValidation, ILogging
{
    private final transient Map<String, String> ruleParams = new HashMap<String, String>();

    protected AbstractValidation()
    {
    	//Empty constructor
    }
    
    public abstract boolean isValid(final String value);

    /**
     * This method adds a rule parameter to the rule definition.
     * Rule parameters are used to make configurable constant values
     * available to the isValid() method.
     * @param name String
     * @param value String
     */
    public void addParam(final String name, final String value)
    {
        if (!StringUtility.isEmpty(name))
        {
            ruleParams.put(name, value);
        }
    }

    /**
     * Returns an iterator of the rule parameters that have been added to the
     * validation.
     * @return Iterator
     */
    protected Iterator<String> getParamNames()
    {
        return ruleParams.keySet().iterator();
    }

    /**
     * Returns the parameter value for the specified parameter name,
     * or null if the parameter name does not exist.
     * @param paramName String
     * @return String
     */
    protected String getParamValue(final String paramName)
    {
        return (String) ruleParams.get(paramName);
    }

    /**
     * This method gives the simplest Regular Expression interface most commonly
     * used by validations. This method takes a regular expression and the value
     * and determines if the value conforms to the rules established by the
     * regular expression.
     *
     * @param regex String
     * @param value String
     * @return boolean
     */
    protected boolean matches(final String regex, final String value)
    {
        final Pattern pattern = Pattern.compile(regex);
        final Matcher matcher = pattern.matcher(value);
        return matcher.matches();
    }
}
