package zen.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import zen.common.Message;
import zen.logging.interfaces.ILogging;
import zen.object.ObjectException;
import zen.object.ObjectUtility;
import zen.string.StringUtility;
import zen.validation.abstracts.AbstractValidation;

/**
 * This is the static single point of entry to the validation layer. This is the
 * class the handles processing and sends back the messages.
 *
 * @author DCano
 *
 */
public final class Validator implements ILogging
{
    /** Used on dependent (child) validations to compare its property with its parent property */
    private final static String PARENT_EQUAL = "PARENT_EQUAL";
    /** Used on dependent (child) validations to compare its property with its parent property */
    private final static String PARENT_NOT_EQUAL = "PARENT_NOT_EQUAL";
    /** Used on parent (first level) validations to determine when child validations should be run */
    private final static String IF_THIS_EMPTY = "IF_THIS_EMPTY";
    /** Used on parent (first level) validations to determine when child validations should be run */
    private final static String IF_THIS_NOT_EMPTY = "IF_THIS_NOT_EMPTY";

    private Validator()
    {
    	super();
    	
    	//Empty constructor
    }
    
    /**
     * This method takes the application name as a string argument, and the
     * Object that the validations will be running against. The method assumes
     * that no optional key was assigned and that only the default validation
     * set will be used. This method passes back a collection of messages the
     * validations may have generated.
     *
     * @param String application - the application name
     * @param Object object - the object to validate
     * @return List of framework Message objects
     */
    public static List<Message> validate(final String application, final Object object)
    {
        return validate(application, null, null, object);
    }

    /**
     * This method takes the application name, optional key, optional validation
     * set, the Object that the validations will be running against. The method
     * assumes that no optional key was assigned and that only the default
     * validation set will be used. This method passes back a collection of
     * messages the validations may have generated.<br>
     * This method exits with warnings if the validation or object mappings
     * cannot be found for the specified application, optkey/object or validationSet.
     *
     * @param String application - the application name
     * @param String optkey - optional mapping key (use null or empty String to ignore)
     * @param String validationSet - optional set (use null or empty String to use the default set)
     * @param Object object - the object to validate
     * @return List of framework Message objects
     */
    public static List<Message> validate(final String application, final String optkey, final String validationSet, final Object object)
    {
        final List<Message> messages = new ArrayList<Message>();

        if (object ==null)
        {
    		return messages;
    	}

        try
        {
            LOG.info(Validator.class, "Get Validation Mapping by application name...");
            final ValidationMapping vmapping = ValidationFactory.getInstance().getValidationMapping(application);
            LOG.info(Validator.class, "Get specific validation mapping for this object[" + object.getClass().getName() + "]...");

            if (vmapping == null)
            {
            	LOG.warn(Validator.class, "Could not find validation mapping for application '"
            			+ (application == null ? "null" : application) + "'");
            }
            else
            {
            	// objects are mapped by their object name if no optkey was
	            // specified in the XML
            	getObjectMapping(vmapping, optkey, validationSet, object, messages);
            }
        }
        catch (MappingNotFoundException exception)
        {
            LOG.error(Validator.class, "MappingNotFoundException: " + exception.getLocalizedMessage(), exception);
        }

        return messages;
    }
    
    private static void getObjectMapping(final ValidationMapping vmapping, final String optkey, final String validationSet, final Object object, final List<Message> messages) throws MappingNotFoundException
    {
    	ObjectMapping omapping = null;
    	
        if (StringUtility.isEmpty(optkey))
        {
            omapping = vmapping.getMapping(object.getClass().getName());
        }
        else
        {
            omapping = vmapping.getMapping(optkey);
        }

        if (omapping == null)
        {
        	LOG.warn(Validator.class, "Could not find object mapping for "
        			+ (StringUtility.isEmpty(optkey) ? "object '"
        					+ object.getClass().getName() : "optkey: " + optkey) + "'");
        }
        else
        {
        	// definition sets are stored in a default set named "" if no set
            // was specified in the XML
        	getDefinitions(omapping, vmapping, validationSet, object, messages);
        }
    }
    
    private static void getDefinitions(final ObjectMapping omapping, final ValidationMapping vmapping, final String validationSet, final Object object, final List<Message> messages)
    {
    	List<ValidationDefinition> definitions = null;
    	
        if (StringUtility.isEmpty(validationSet))
        {
            definitions = omapping.getDefinitions();
        }
        else
        {
            definitions = omapping.getDefinitions(validationSet);
        }

        if (definitions != null)
        {
            LOG.info(Validator.class, "Run the validation rules....");
            processDefinitions(object, vmapping, definitions, messages, null);
        }
    }

    /**
     * Processes all validations for s specific mapping and validation set.
     *
     * @param object
     * @param mapping
     * @param definitions
     * @param messages
     * @param parentPropertyName
     */
    protected static void processDefinitions(final Object object, final ValidationMapping mapping, final List<ValidationDefinition> definitions, final List<Message> messages, final ValidationDefinition parentDefinition)
    {
        for (ValidationDefinition definition: definitions)
        {
        	String property = null;
        	
            //if (!StringUtility.isEmpty(definition.getProperty())) {
            if (parentDefinition == null)
            {
                property = definition.getProperty();
            }
            else
            {
                property = parentDefinition.getProperty();
            }

            final String value = getValue(object, property);
            executeDefinition(object, mapping, definition, messages, property, value);
        }
    }

    /**
     * Calls the isValid() method on the validation implementation.
     *
     * @param object
     * @param mapping
     * @param definition
     * @param messages
     * @param parentPropName
     */
    private static void executeDefinition(final Object object, final ValidationMapping mapping, final ValidationDefinition definition, final List<Message> messages, final String parentPropName, final String parentPropValue)
    {
        try
        {
        	final String property = getProperty(definition, parentPropName);
            final String value = getValue(object, property);
            final Message message = new Message(definition.getMessage(), definition.getParameters());
            final boolean proceed = getProceed(value, definition.getRuleName(), parentPropValue, mapping, messages, message);

            if (proceed)
            {
                final List<ValidationDefinition> dependencies = definition.getDependencies();
                processDefinitions(object, mapping, dependencies, messages, definition);
            }
        }
        catch (RuleNotFoundException e)
        {
            LOG.error(Validator.class, "This Rule [" + definition.getRuleName() + "] does not exist in the rules mappings", e);
        }
    }
    
    private static boolean getProceed(final String value, final String ruleName, final String parentPropValue, final ValidationMapping mapping, final List<Message> messages, final Message message) throws RuleNotFoundException
    {
    	boolean proceed = true;

        if (ruleName.equals(IF_THIS_EMPTY))
        {
        	proceed = StringUtility.isEmpty(value);
        }
        else if (ruleName.equals(IF_THIS_NOT_EMPTY))
        {
        	proceed = !StringUtility.isEmpty(value);
        }
        else
        {
            // validate
            if (ruleName.equals(PARENT_EQUAL))
            {
                if (!value.equals(parentPropValue))
                {
                    proceed = false;
                    messages.add(message);
                }
            }
            else if (ruleName.equals(PARENT_NOT_EQUAL))
            {
                if (value.equals(parentPropValue))
                {
                    proceed = false;
                    messages.add(message);
                }
            }
            else
            {
                // normal validation
                final AbstractValidation validation = mapping.getRule(ruleName);

                if (!validation.isValid(value))
                {
                    proceed = false;
                    messages.add(message);
                }
            }
        }
        
        return proceed;
    }
    
    private static String getProperty(final ValidationDefinition definition, final String parentPropName)
    {
    	if (StringUtility.isEmpty(definition.getProperty()))
        {
        	return parentPropName;
        }
        else
        {
        	return definition.getProperty();
        }
    }

	/*
	 * Ugly hack.  God forgive me.
	 * So were dynaforms a great idea or what.
	 * I climb the class hierarchy so that we don't create a
	 * direct dependency on DynaForms in here.
	 */
    @SuppressWarnings("unchecked")
	private static String getValue(final Object object, final String property)
    {
        Class clazz = object.getClass();
        String value = null;

        try 
        {
			while (clazz != null) 
			{
			    if (clazz.getName().equals("org.apache.struts.action.DynaActionForm"))
			    {
			    	final Map<String, String> map = (Map<String, String>) ObjectUtility.getMethodValue(object, "map");
			    	value = (String) map.get(property);
			    	break;
			    }

			    clazz = clazz.getSuperclass();
			}

			if (clazz == null)
			{
				// not a dynaform, so be normal now
				value = (String) ObjectUtility.getMethodValue(object, property);
			}
		} 
        catch (ObjectException e) 
        {
            LOG.error(Validator.class, "Error getting value for Property [" + property + "]", e);
		}

        return value;
    }
}
