package zen.validation;

import java.util.HashMap;
import java.util.Map;

import zen.common.AbstractFactory;
import zen.configuration.ConfigurationException;
import zen.configuration.ConfigurationReader;


/**
 * This is the Factory class that loads the configurations and holds the validation mappings
 * for specific objects for specific applications. This approach was taken as applications,
 * which may be functionally the same may have different validation rules based on the
 * application. For instance, a zen application for two clients may be functionally
 * the same, but have different requirements for validations based on the clients' needs.
 *
 * @author DCano
 *
 */
public final class ValidationFactory extends AbstractFactory 
{
	private static final long serialVersionUID = 1301702902802834345L;
	private static ValidationFactory instance;
    private final transient Map<String, ValidationMapping> mappings = new HashMap<String, ValidationMapping>();

    private ValidationFactory()
    {
    	super();
    	
        try
        {
            final Configurations configurations = (Configurations) ConfigurationReader.loadConfigurations(Configurations.class);
            
            for (Configuration configuration : configurations.getConfigurations())
            {
                mappings.put(configuration.getName(), getValidationMapping(configuration));
            }
        }
        catch (ConfigurationException e)
        {
            LOG.error(ValidationFactory.class, "ConfigurationException: " + e.getLocalizedMessage(), e);
        }
    }
    
    private ValidationMapping getValidationMapping(final Configuration configuration)
    {
    	return new ValidationMapping(configuration.getFile());
    }

    public static ValidationFactory getInstance()
    {
        if (instance == null)
        {
            instance = new ValidationFactory();
        }

        return instance;
    }

    public ValidationMapping getValidationMapping(final String application)
    {
        if (mappings.containsKey(application))
        {
            return (ValidationMapping) mappings.get(application);
        }

        return null;
    }
}
