package zen.validation;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import zen.logging.interfaces.ILogging;
import zen.xml.XmlException;

/**
 * Contains sets of validation definition collections for the Object.
 * A default set is defined as any validation tags that are not nested
 * within a set tag, or are within a set tag having an empty string as
 * its name attribute.
 *
 * @author DCano
 *
 */
public class ObjectMapping implements Serializable, ILogging
{
	private static final long serialVersionUID = -6943873183844776182L;
    private String object = null;
    private final transient Map<String, ArrayList<ValidationDefinition>> definitions = new HashMap<String, ArrayList<ValidationDefinition>>();

    /**
     * Constructor for an object mapping using the .
     * Creates the ValidationDefinition lists for the default validation set
     * (that is, any validation tags not nested within a set tag)
     * and any other validation sets defined in the validations XML.
     * The object parameter should be the fully qualified class name of the object to be validated.
     * The resource is the relative path to the matching XML validations definition file.
     * @param String object
     * @param String resource
     */
    public ObjectMapping(final String object, final String resource)
    {
        this.object = object;

        try
        {
        	MappingUtility.getDefinitions(resource, definitions);
        }
        catch (XmlException e)
        {
            LOG.error(ObjectMapping.class, e.toString(), e);
        }
    }
    
    /**
     * If the specified validation set exists, returns True, otherwise returns False.
     * To check for the default validation set, pass in an empty String.
     * @param String set
     * @return boolean
     */
    public boolean hasDefinitions(final String set)
    {
        return definitions.containsKey(set);
    }

    /**
     * Returns a list of validation definitions for the default validation set.
     * If no default set has been defined, null is returned.
     * @return List
     */
    public List<ValidationDefinition> getDefinitions()
    {
        return getDefinitions("");
    }

    /**
     * Returns a list of ValidationDefinition objects for the specified validation set.
     * @param String set
     * @return List of ValidationDefinition objects
     */
    public List<ValidationDefinition> getDefinitions(final String set)
    {
        return (List<ValidationDefinition>) definitions.get(set);
    }

    /**
     * Returns the object identifier that was set
     * (typically the fully-qualified class name of the object to validate).
     * @return String
     */
    public String getObject()
    {
        return object;
    }

    /**
     * Sets the object identifier that was set
     * (typically the fully-qualified class name of the object to validate).
     * @param String object
     */
    public void setObject(final String object)
    {
        this.object = object;
    }
}
