package zen.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;


import zen.object.ObjectException;
import zen.object.ObjectUtility;
import zen.string.StringUtility;
import zen.validation.abstracts.AbstractValidation;
import zen.xml.XmlDocument;
import zen.xml.XmlException;
import zen.xml.XmlNode;

public final class MappingUtility
{
	private final static String SET = "set";
	private final static String PROPERTY = "property";
    private final static String MESSAGE = "message";
    private final static String PARAMETERS = "parameters";
    private final static String VALIDATION = "validation";
    private final static String RULE = "rule";
    private final static String RULE_PARAM = "param";
    private final static String NAME = "name";
    private final static String TYPE = "type";
    private final static String VALUE = "value";
    private final static String OPTKEY = "optkey";
    private final static String OBJECT = "object";
    private final static String RESOURCE = "resource";
    
	private MappingUtility()
	{
		super();
		
		//Empty constructor
	}
	
	public static void getDefinitions(final String resource, final Map<String, ArrayList<ValidationDefinition>> definitions) throws XmlException
	{
		final XmlDocument document = new XmlDocument(resource);
    	final XmlNode root = document.getRoot();

    	List<XmlNode> nodes = root.getChildren(VALIDATION);
        
        addDefinitions(definitions, "", nodes);

        // now do the explicitly defined sets
        final List<XmlNode> sets = root.getChildren(SET);
        
        for (int i = 0; i < sets.size(); i++)
        {
            final XmlNode setNode = (XmlNode) sets.get(i);
            final String set = setNode.getAttribute("key");
            nodes = setNode.getChildren(VALIDATION);
            addDefinitions(definitions, set, nodes);
        }
	}
	
	public static void addDefinitions(final Map<String, ArrayList<ValidationDefinition>> definitions, final String set, final List<XmlNode> nodes)
	{
		for (XmlNode node : nodes)
    	{
    		addDefinition(definitions, set, getValidationDefinition(node));
    	}
	}
	
	private static ValidationDefinition getValidationDefinition(final XmlNode node)
    {
    	final ValidationDefinition definition = new ValidationDefinition();
        definition.setProperty(node.getAttribute(PROPERTY));
        definition.setRuleName(node.getAttribute(RULE));
        definition.setMessage(node.getAttribute(MESSAGE));

        final String parameters = node.getAttribute(PARAMETERS);

        if (!StringUtility.isEmpty(parameters))
        {
            definition.setParameters(StringUtility.split(parameters, ","));
        }

        for (XmlNode child : node.getChildren(VALIDATION))
        {
        	definition.addDependency(getValidationDefinition(child));
        }

        return definition;
    }
	
	public static void addDefinition(final Map<String, ArrayList<ValidationDefinition>> definitions, final ValidationDefinition definition)
    {
        addDefinition(definitions, "", definition);
    }

    /**
     * Adds a definition to the specified validation set.
     * @param set
     * @param definition
     */
    public static void addDefinition(final Map<String, ArrayList<ValidationDefinition>> definitions, final String set, final ValidationDefinition definition)
    {
        if (definitions.containsKey(set))
        {
            final ArrayList<ValidationDefinition> defs = (ArrayList<ValidationDefinition>) definitions.get(set);
            defs.add(definition);
        }
        else
        {
            final ArrayList<ValidationDefinition> defs = new ArrayList<ValidationDefinition>();
            defs.add(definition);
            definitions.put(set, defs);
        }
    }
    
    public static void addRules(final Map<String, AbstractValidation> rules, final List<XmlNode> nodes) throws ObjectException
    {
    	for (XmlNode node : nodes)
    	{
            final String name = node.getAttribute(NAME);
            final String type = node.getAttribute(TYPE);
            final AbstractValidation validation = (AbstractValidation) ObjectUtility.createInstance(type);

            // add any rule parameters
            for (XmlNode paramNode : node.getChildren(RULE_PARAM))
            {
                final String paramName = paramNode.getAttribute(NAME);
                final String paramValue = paramNode.getAttribute(VALUE);
                validation.addParam(paramName, paramValue);
            }

            rules.put(name, validation);
        }
    }
    
    public static void addMappings(final Map<String, ObjectMapping> objects, final List<XmlNode> nodes)
    {
    	for (XmlNode node : nodes)
        {
    		addMapping(objects, node);
        }
    }
    
    private static void addMapping(final Map<String, ObjectMapping> objects, final XmlNode node)
    {
    	final String object = node.getAttribute(OBJECT);
        
        String optkey = node.getAttribute(OPTKEY);

        if (StringUtility.isEmpty(optkey))
        {
            optkey = object;
        }

        final String resource = node.getAttribute(RESOURCE);
        final ObjectMapping mapping = new ObjectMapping(object, resource);

        objects.put(optkey, mapping);
    }
}
