package zen.scm.implementations.svn;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import zen.scm.CommandException;
import zen.scm.abstracts.AbstractLogCommand;
import zen.scm.constants.Modification;
import zen.scm.objects.ChangeObject;
import zen.scm.objects.ProfileObject;
import zen.scm.objects.RevisionObject;
import zen.string.StringUtility;
import zen.system.Feedback;
import zen.xml.XmlDocument;
import zen.xml.XmlException;
import zen.xml.XmlNode;

public class SVNLogCommand extends AbstractLogCommand
{
	private final static String LOG = "log --xml";
	private final static String STOP_ON_COPY = "--stop-on-copy";
	private final static String USE_MERGE_HISTORY = "--use-merge-history";
	private final static String CHANGED = "changed";
	private final static String REVISION = "-r";
	
	public SVNLogCommand(final ProfileObject profile)
	{
		super(profile);
	}
	
	@Override
	public List<RevisionObject> revisions(final String path, final boolean stopOnCopy, final boolean useMergeHistory) throws CommandException
	{
		final StringBuffer command = new StringBuffer();
		command.append(getLogCommand(stopOnCopy, useMergeHistory));
		command.append(SvnConstants.SPACE);
		command.append(path);
		return getRevisions(path, command.toString());
	}

	@Override
	public List<RevisionObject> revisions(final String path, final Long revision, final boolean stopOnCopy, final boolean useMergeHistory) throws CommandException
	{
		final StringBuffer command = new StringBuffer();
		command.append(getLogCommand(stopOnCopy, useMergeHistory));
		command.append(SvnConstants.SPACE);
		command.append(path);
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(revision);
		return getRevisions(path, command.toString());
	}

	@Override
	public List<RevisionObject> revisions(final String path, final Long fromRevision, final Long toRevision, final boolean stopOnCopy, final boolean useMergeHistory) throws CommandException
	{
		final StringBuffer command = new StringBuffer();
		command.append(getLogCommand(stopOnCopy, useMergeHistory));
		command.append(SvnConstants.SPACE);
		command.append(path);
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(fromRevision);
		command.append(SvnConstants.COLON);
		command.append(toRevision);
		return getRevisions(path, command.toString());
	}
	
	@Override
	public List<RevisionObject> revisions(final String path, final Date date, final boolean stopOnCopy, final boolean useMergeHistory) throws CommandException
	{
		final StringBuffer command = new StringBuffer();
		command.append(getLogCommand(stopOnCopy, useMergeHistory));
		command.append(SvnConstants.SPACE);
		command.append(path);
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(SvnUtility.getDateArguments(date, date));
		
		final List<RevisionObject> revisions =  getRevisions(path, command.toString());
		
		adjustLogs(revisions);
		
		return revisions;
	}
	
	@Override
	public List<RevisionObject> revisions(final String path, final Date fromDate, final Date toDate, final boolean stopOnCopy, final boolean useMergeHistory) throws CommandException
	{
		final StringBuffer command = new StringBuffer();
		command.append(getLogCommand(stopOnCopy, useMergeHistory));
		command.append(SvnConstants.SPACE);
		command.append(path);
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(SvnUtility.getDateArguments(fromDate, toDate));
		
		final List<RevisionObject> logs = getRevisions(path, command.toString());
		
		adjustLogs(logs);
		
		return logs;
	}
	
	public List<ChangeObject> changes(final RevisionObject revision) throws CommandException
	{
		final String hpath = getProfile().getHttp();
		final String ppath = getProfile().getPath().replace('\\', '/');
		final String rpath = revision.getPath().replace('\\', '/');
		final Long rev = revision.getRevision();
		
		final StringBuffer command = new StringBuffer();
		command.append(getLogCommand(true, false));
		command.append(SvnConstants.SPACE);
		command.append(SvnConstants.VERBOSE);
		command.append(SvnConstants.SPACE);
		command.append(hpath);
		command.append('/');
		command.append(rpath);
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(revision.getRevision());
		
		final List<XmlNode> entries = getEntries(command.toString());
		
		final List<ChangeObject> changes = new ArrayList<ChangeObject>(0);
		
		if (entries != null && entries.size() == 1)
		{
			changes.addAll(getChanges(entries.get(0)));
			getChanged(ppath, rev, changes);
		}
		
		return changes;
	}
	
	private String getLogCommand(final boolean stopOnCopy, final boolean useMergeHistory)
	{
		final StringBuffer command = new StringBuffer();
		
		command.append(LOG);
		
		if (stopOnCopy)
		{
			command.append(SvnConstants.SPACE);
			command.append(STOP_ON_COPY);
		}
		
		if (useMergeHistory)
		{
			command.append(SvnConstants.SPACE);
			command.append(USE_MERGE_HISTORY);
		}
		
		return command.toString();
	}
	
	private void adjustLogs(final List<RevisionObject> revisions)
	{
		if (!revisions.isEmpty())
		{
			revisions.remove(0);
		}
	}
	
	private List<XmlNode> getEntries(final String command) throws CommandException
	{
		try
		{
			final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command);
			
			if (feedback.isError())
			{
				throw new CommandException(getErrorMessage(feedback));
			}
			else
			{
				final XmlDocument document = feedback.getInputAsXmlDocument();
				return document.getRoot().getChildren("logentry");
			}
		}
		catch (XmlException exception)
		{
			throw new CommandException(exception);
		}
	}

	private List<RevisionObject> getRevisions(final String path, final String command) throws CommandException
	{
		final List<RevisionObject> revisions = new ArrayList<RevisionObject>();
		final List<XmlNode> entries = getEntries(command);
		
		for (XmlNode entry : entries)
		{
			revisions.add(getRevision(path, entry));
		}

		return revisions;
	}
	
	private RevisionObject getRevision(final String path, final XmlNode entry) throws CommandException
	{
		final RevisionObject revision = new RevisionObject();
		revision.setPath(path.substring(getProfile().getHttp().length()));
		revision.setRevision(Long.valueOf(entry.getAttribute("revision")));
		revision.setAuthor(entry.getChild("author").getValue());
		revision.setTimestamp(SvnUtility.getSVNFormattedDate(entry.getChild("date").getValue()));
		revision.setMessage(entry.getChild("msg").getValue());
		return revision;
	}
	
	private List<ChangeObject> getChanges(final XmlNode entry) throws CommandException
	{
		final List<ChangeObject> changes = new ArrayList<ChangeObject>();
		final List<XmlNode> nodes = entry.getChild("paths").getChildren("path");
		
		for (XmlNode node : nodes)
		{
			changes.add(getChange(node));
		}
		
		return changes;
	}
	
	private ChangeObject getChange(final XmlNode node) throws CommandException
	{
		final ChangeObject change = new ChangeObject();
		change.setKind(SvnUtility.getKind(node.getAttribute("kind")));
		change.setChange(SvnUtility.getChange(node.getAttribute("action")));
		change.setPath(node.getValue());
		
		if (!StringUtility.isEmpty(node.getAttribute("copyfrom-path")) && !StringUtility.isEmpty(node.getAttribute("copyfrom-rev")))
		{
			change.setFromPath(node.getAttribute("copyfrom-path"));
			change.setFromRevision(Long.valueOf(node.getAttribute("copyfrom-rev")));	
		}
		
		return change;
	}

	private void getChanged(final String path, final Long revision, final List<ChangeObject> changes) throws CommandException
	{
		if (revision != null)
		{
			final StringBuffer command = new StringBuffer();
			command.append(CHANGED);
			command.append(SvnConstants.SPACE);
			command.append(REVISION);
			command.append(SvnConstants.SPACE);
			command.append(revision);
			command.append(SvnConstants.SPACE);
			command.append(path);
			
			final Feedback feedback = SvnUtility.getSvnLookCommand(command.toString());
			
			if (feedback.getInput() != null && feedback.getInput().size() > 0)
			{
				for (String line : feedback.getInput())
				{
					final String mod = line.substring(0, 2).trim();
					final String fil = line.substring(2).trim();
					
					for (ChangeObject change : changes)
					{
						setChanged(change, mod, fil);
					}
				}
			}
		}
	}
	
	private void setChanged(final ChangeObject change, final String mod, final String fil)
	{
		if (change.getPath().indexOf(fil) > -1)
		{
			if (mod.equals(Modification.FILE))
			{
				change.setModifiedFile(true);
			}
			else if (mod.equals(Modification.PROPERTY))
			{
				change.setModifiedProperty(true);
			}
			else if (mod.equals(Modification.FILE_AND_PROPERTY))
			{
				change.setModifiedFile(true);
				change.setModifiedProperty(true);
			}
		}
	}
}
