package zen.scm.implementations.svn;

import java.util.ArrayList;
import java.util.List;

import zen.scm.CommandException;
import zen.scm.abstracts.AbstractDiffCommand;
import zen.scm.enums.Change;
import zen.scm.objects.ChangeObject;
import zen.scm.objects.ProfileObject;
import zen.system.Feedback;
import zen.xml.XmlDocument;
import zen.xml.XmlException;
import zen.xml.XmlNode;

public class SVNDiffCommand extends AbstractDiffCommand
{
	private final static String DIFF_COMMAND = "diff";
	private final static String XML = "--xml";
	private final static String SUMMARY = "--summarize";
	private final static String REVISION = "-r";
	private final static String AT_SYMBOL = "@";
	
	public SVNDiffCommand(final ProfileObject profile)
	{
		super(profile);
	}
	
	@Override
	public List<String> diff(final String path1, final Long revision1, final Long revision2) throws CommandException
	{
		final StringBuffer command = new StringBuffer(20);
		command.append(getCommand(false));
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(revision1);
		command.append(SvnConstants.COLON);
		command.append(revision2);
		command.append(SvnConstants.SPACE);
		command.append(path1);
		
		final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command.toString());
		return process(feedback);
	}

	@Override
	public List<String> diff(final String path1, final Long revision1, final String path2, final Long revision2) throws CommandException
	{
		final StringBuffer command = new StringBuffer(20);
		command.append(getCommand(false));
		command.append(SvnConstants.SPACE);
		command.append(path1);
		command.append(AT_SYMBOL);
		command.append(revision1);
		command.append(SvnConstants.SPACE);
		command.append(path2);
		command.append(AT_SYMBOL);
		command.append(revision2);
		
		final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command.toString());
		return process(feedback);
	}

	@Override
	public List<String> diff(final String path1, final String path2) throws CommandException
	{
		final StringBuffer command = new StringBuffer(20);
		command.append(getCommand(false));
		command.append(SvnConstants.SPACE);
		command.append(path1);
		command.append(AT_SYMBOL);
		command.append(SvnConstants.HEAD);
		command.append(SvnConstants.SPACE);
		command.append(path2);
		command.append(AT_SYMBOL);
		command.append(SvnConstants.HEAD);
		
		final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command.toString());
		return process(feedback);
	}
	
	@Override
	public List<ChangeObject> summarize(final String path1, final Long revision1, final Long revision2) throws CommandException
	{
		final StringBuffer command = new StringBuffer(20);
		command.append(getCommand(true));
		command.append(SvnConstants.SPACE);
		command.append(REVISION);
		command.append(SvnConstants.SPACE);
		command.append(revision1);
		command.append(SvnConstants.COLON);
		command.append(revision2);
		command.append(SvnConstants.SPACE);
		command.append(path1);
		
		final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command.toString());
		return getSummary(feedback);
	}

	@Override
	public List<ChangeObject> summarize(final String path1, final Long revision1, final String path2, final Long revision2) throws CommandException
	{
		final StringBuffer command = new StringBuffer(20);
		command.append(getCommand(true));
		command.append(SvnConstants.SPACE);
		command.append(path1);
		command.append(AT_SYMBOL);
		command.append(revision1);
		command.append(SvnConstants.SPACE);
		command.append(path2);
		command.append(AT_SYMBOL);
		command.append(revision2);
		
		final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command.toString());
		return getSummary(feedback);
	}

	@Override
	public List<ChangeObject> summarize(final String path1, final String path2) throws CommandException
	{
		final StringBuffer command = new StringBuffer(20);
		command.append(getCommand(true));
		command.append(SvnConstants.SPACE);
		command.append(path1);
		command.append(AT_SYMBOL);
		command.append(SvnConstants.HEAD);
		command.append(SvnConstants.SPACE);
		command.append(path2);
		command.append(AT_SYMBOL);
		command.append(SvnConstants.HEAD);
		
		final Feedback feedback = SvnUtility.runSvnCommand(getProfile(), command.toString());
		return getSummary(feedback);
	}
	
	private String getCommand(final boolean summarize)
	{
		String command = null;
		
		if (summarize)
		{
			command = DIFF_COMMAND + SvnConstants.SPACE + XML + SvnConstants.SPACE + SUMMARY;
		}
		else 
		{
			command = DIFF_COMMAND;
		}
		
		return command;
	}
	
	private List<String> process(final Feedback feedback) throws CommandException
	{
		if (feedback.isError())
		{
			throw new CommandException(getErrorMessage(feedback));
		}
		else
		{
			return feedback.getInput();
		}
	}
	
	private List<ChangeObject> getSummary(final Feedback feedback) throws CommandException
	{
		try
		{
			final List<ChangeObject> changes = new ArrayList<ChangeObject>();
			
			if (feedback.isError())
			{
				throw new CommandException(getErrorMessage(feedback));
			}
			else
			{
				final XmlDocument document = feedback.getInputAsXmlDocument();
				final List<XmlNode> paths = document.getNode("paths").getChildren("path");
				
				for (XmlNode path : paths)
				{
					changes.add(getChange(path));
				}
			}
			
			return changes;
		}
		catch (XmlException exception)
		{
			throw new CommandException(exception);
		}
	}
	
	private ChangeObject getChange(final XmlNode path)
	{
		final ChangeObject change = new ChangeObject();
		change.setPath(path.getValue());
		setChange(change, path);
		change.setKind(SvnUtility.getKind(path.getAttribute("kind")));
		return change;
	}
	
	private void setChange(final ChangeObject change, final XmlNode path)
	{
		final String item = path.getAttribute("item");
		
		if ("added".equals(item))
		{
			change.setChange(Change.ADDED);
			change.setModifiedFile(true);
		}
		else if ("deleted".equals(item))
		{
			change.setChange(Change.DELETED);
			change.setModifiedFile(true);
		}
		else if ("modified".equals(item))
		{
			change.setChange(Change.MODIFIED);
			change.setModifiedFile(true);
		}
		else if ("none".equals(item))
		{
			change.setModifiedFile(false);
		}
		
		final String props = path.getAttribute("props");
		
		if ("added".equals(props))
		{
			change.setChange(Change.ADDED);
			change.setModifiedProperty(true);
		}
		else if ("deleted".equals(props))
		{
			change.setChange(Change.DELETED);
			change.setModifiedProperty(true);
		}
		else if ("modified".equals(props))
		{
			change.setChange(Change.MODIFIED);
			change.setModifiedProperty(true);
		}
		else if ("none".equals(props))
		{
			change.setModifiedProperty(false);
		}
	}
}
