package zen.scm;

import java.util.HashMap;
import java.util.Map;

import zen.logging.interfaces.ILogging;
import zen.scm.abstracts.AbstractCatCommand;
import zen.scm.abstracts.AbstractCheckoutCommand;
import zen.scm.abstracts.AbstractCommandFactory;
import zen.scm.abstracts.AbstractDiffCommand;
import zen.scm.abstracts.AbstractInfoCommand;
import zen.scm.abstracts.AbstractListCommand;
import zen.scm.abstracts.AbstractLockCommand;
import zen.scm.abstracts.AbstractLogCommand;
import zen.scm.enums.Scm;
import zen.scm.implementations.svn.SVNCommandFactory;
import zen.scm.objects.ProfileObject;

public final class Commands implements ILogging
{
	private final static Map<Scm, AbstractCommandFactory> FACTORIES = new HashMap<Scm, AbstractCommandFactory>();
	
	static
	{
		FACTORIES.put(Scm.SVN, new SVNCommandFactory());
	}
	
	private Commands()
	{
		super();
		
		//Empty constructor
	}
	
	public static AbstractCatCommand getContentCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getContentCommand(profile);
	}
	
	public static AbstractDiffCommand getDifferenceCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getDifferenceCommand(profile);
	}
	
	public static AbstractListCommand getListCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getListCommand(profile);
	}
	
	public static AbstractLockCommand getLockCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getLockCommand(profile);
	}
	
	public static AbstractLogCommand getLogCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getLogCommand(profile);
	}
	
	public static AbstractInfoCommand getInfoCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getInfoCommand(profile);
	}
	
	public static AbstractCheckoutCommand getCheckoutCommand(final ProfileObject profile) throws CommandException
	{
		return getFactory(profile).getCheckoutCommand(profile);
	}

	public static AbstractCommandFactory getFactory(final ProfileObject profile) throws CommandException
	{
		final AbstractCommandFactory factory = (AbstractCommandFactory) getFactory(profile.getScm());
		
		if (factory != null)
		{
			return factory;
		}
		
		throw new CommandException("No factory found for SCM: " + profile.getScm().toString());
	}
	
	private static AbstractCommandFactory getFactory(final Scm scm)
	{
		AbstractCommandFactory factory = null;
		
		if (FACTORIES.containsKey(scm))
		{
			factory = (AbstractCommandFactory) FACTORIES.get(scm);
		}
		
		return factory;
	}
}
