package zen.pdf;

import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;

import org.jfree.chart.JFreeChart;

import zen.classpath.ClasspathUtility;
import zen.logging.interfaces.ILogging;

import com.itextpdf.text.BadElementException;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Font;
import com.itextpdf.text.Image;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.DefaultFontMapper;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfDocument;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfTemplate;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.text.pdf.draw.LineSeparator;

public class PDFPage implements ILogging
{
	private transient final PdfWriter writer;
	private transient final Document document;
	private transient final PdfDocument pdfDocument;
	private transient final PdfContentByte content;
	
	protected PDFPage(final PdfWriter writer, final Document document)
	{
		this.writer = writer;
		this.document = document;
		this.content = writer.getDirectContentUnder();
		this.pdfDocument = content.getPdfDocument();
	}
	
	public float getWidth()
	{
		return content.getPdfDocument().getPageSize().getWidth();
	}
	
	public float getHeight()
	{
		return content.getPdfDocument().getPageSize().getHeight();
	}
	
	public PDFHeader getPDFHeader()
	{
		return new PDFHeader(writer);
	}
	
	public PDFFooter getPDFFooter()
	{
		return new PDFFooter(writer);
	}
	
	public void writeContent(final String text, final String fontName, final BaseColor fontColor, final float fontSize, final PDFAlign alignment, final int xPosition, final int yPosition) throws DocumentException, IOException
	{
		content.beginText();
		content.setColorFill(fontColor);
		content.setFontAndSize(BaseFont.createFont(fontName, "UTF-8", true), fontSize);
		content.showTextAligned(alignment.value(), text, xPosition, yPosition, 0);
		content.endText();
	}
	
	public void writeContent(final String text, final Font font, final PDFAlign alignment, final int xPosition, final int yPosition) throws DocumentException, IOException
	{
		content.beginText();
		content.setColorFill(font.getColor());
		content.setFontAndSize(BaseFont.createFont(font.getFamilyname(), "UTF-8", true), font.getSize());
		content.showTextAligned(alignment.value(), text, xPosition, yPosition, 0);
		content.endText();
	}
	
	public void writeText(final String text) throws DocumentException
	{
		document.add(new Chunk(text));
	}
	
	public void writeText(final String text, final Font font) throws DocumentException
	{
		document.add(new Chunk(text, font));
	}
	
	public void writeTextLine(final String text) throws DocumentException
	{
		document.add(new Chunk(text));
		document.add(Chunk.NEWLINE);
	}
	
	public void writeTextLine(final String text, final Font font) throws DocumentException
	{
		document.add(new Chunk(text, font));
		document.add(Chunk.NEWLINE);
	}
	
	public void writeLineSeparator() throws DocumentException
	{
		document.add(new LineSeparator());
		document.add(Chunk.NEWLINE);
	}
	
	public void writeLineSeparator(final BaseColor color) throws DocumentException
	{
		document.add(new LineSeparator(1, 100, color, LineSeparator.ALIGN_TOP, -2));
		document.add(Chunk.NEWLINE);
	}

	public void writeLineSeparator(final float lineWidth, final float percentage, final BaseColor color, final int alignment, final float offset) throws DocumentException
	{
		document.add(new LineSeparator(lineWidth, percentage, color, alignment, offset));
		document.add(Chunk.NEWLINE);
	}
	
	public PDFTable newTable(final int numOfColumns, final float width)
	{
		return new PDFTable(new PdfPTable(numOfColumns), width);
	}
	
	public void writeTable(final PDFTable table)
	{
		table.getTable().writeSelectedRows(0, table.getTable().size(), document.leftMargin(), pdfDocument.getVerticalPosition(true), writer.getDirectContent());
	}
	
	public void writeImage(final String imagePath, final float scalePercent, final int xPosition, final int yPosition, final PDFAlign alignment) throws IOException, BadElementException, DocumentException
	{
		final Image image = Image.getInstance(ClasspathUtility.getUrlPath(imagePath));
		image.scalePercent(scalePercent);
		image.setAbsolutePosition(xPosition, yPosition);
		image.setAlignment(alignment.value());
		content.addImage(image);
	}
	
	public void writeImage(final String imagePath, final float scalePercentX, final float scalePercentY, final int xPosition, final int yPosition, final PDFAlign alignment) throws IOException, BadElementException, DocumentException
	{
		final Image image = Image.getInstance(ClasspathUtility.getUrlPath(imagePath));
		image.scalePercent(scalePercentX, scalePercentY);
		image.setAbsolutePosition(xPosition, yPosition);
		image.setAlignment(alignment.value());
		content.addImage(image);
	}
	
	public void writeChart(final JFreeChart chart, final float width, final float height) throws DocumentException
	{
		final PdfContentByte canvas = writer.getDirectContent();
		final PdfTemplate template = canvas.createTemplate(width, height);
		final Graphics2D graphics = template.createGraphics(width, height, new DefaultFontMapper());
		final Rectangle2D rectangle = new Rectangle2D.Double(0, 0, width, height);
		
		chart.draw(graphics, rectangle);
		graphics.dispose();

		final Image image = Image.getInstance(template);
		image.setAlignment(Element.ALIGN_CENTER);
		
		document.add(image);
		document.add(Chunk.NEWLINE);
	}
	
	public void writeChart(final JFreeChart chart, final float width, final float height, final int alignment) throws DocumentException
	{
		final PdfContentByte canvas = writer.getDirectContent();
		final PdfTemplate template = canvas.createTemplate(width, height);
		final Graphics2D graphics = template.createGraphics(width, height, new DefaultFontMapper());
		final Rectangle2D rectangle = new Rectangle2D.Double(0, 0, width, height);
		
		chart.draw(graphics, rectangle);
		graphics.dispose();

		final Image image = Image.getInstance(template);
		image.setAlignment(alignment);
		
		document.add(image);
		document.add(Chunk.NEWLINE);
	}
}
