package zen.object;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import zen.string.StringUtility;
import zen.utility.Utility;

public final class ObjectUtility extends Utility
{
    private static final long serialVersionUID = 8250949864352986524L;
    private static final String GET = "get";
    private static final String SET = "set";
    private static final String PERIOD = ".";

    private ObjectUtility()
    {
    	super();
    	
    	//Empty constructor
    }
    
    public static Object createInstance(final String instance) throws ObjectException
    {
        try
        {
            return Class.forName(instance).newInstance();
        }
        catch (IllegalAccessException exception)
        {
            throw new ObjectException("Class [" + instance + "] could not be accessed.", exception);
        }
        catch (InstantiationException exception)
        {
            throw new ObjectException("Class [" + instance + "] could not be instantiated.", exception);
        }
        catch (ClassNotFoundException exception)
        {
            throw new ObjectException("Class [" + instance + "] could not be found on the classpath.", exception);
        }
        catch (Exception exception)
        {
            throw new ObjectException("Exception [" + exception.toString() + "] occured while trying to create an instance of Class [" + instance + "]", exception);
        }
    }

	public static void setMethodValue(final Object obj, final Object[] values, final String fullPath) throws ObjectException
    {
        try
        {
            Method method = null;
            final String[] elements = StringUtility.split(fullPath, PERIOD);
            final Object object = obj;

            if (elements.length == 1)
            {
            	final Class superClass = object.getClass();
                method = getMethod(superClass, fullPath, false);

                if (method != null)
                {
                    method.invoke(object, values);
                }

                //If method does not exist on this class, it must exist on a parent
                setSuperClassMethod(superClass, object, values, fullPath);
            }
            else
            {
            	//If method does not exist on this class, it must exist on a child
            	setChildClassMethod(elements, object, values);
            }
        }
        catch (InvocationTargetException exception)
        {
            throw new ObjectException(fullPath + " could not be invoked on Object [" + obj.getClass().getName() + "]", exception);
        }
        catch (IllegalAccessException exception)
        {
            throw new ObjectException(fullPath + " could not be accessed on Object [" + obj.getClass().getName() + "]", exception);
        }
        catch (Exception exception)
        {
            throw new ObjectException("Some error occured when trying to access/invoke Method [" + fullPath + "] on Object [" + obj.getClass().getName() + "]", exception);
        }
    }
    
    private static void setSuperClassMethod(final Class child, final Object object, final Object[] values, final String fullPath) throws InvocationTargetException, IllegalAccessException
    {
    	Class superClass = child;
    	
    	while ((superClass = superClass.getSuperclass()) != null)
        {
            final Method method = getMethod(superClass, fullPath, false);

            if (method != null)
            {
                method.invoke(object, values);
            }
        }
    }
    
    private static void setChildClassMethod(final String[] elements, final Object object, final Object[] values) throws InvocationTargetException, IllegalAccessException
    {
    	Object thisObject = object;
    	
    	for (int i = 0; i < elements.length; i++)
        {
            final Method method = getMethod(thisObject.getClass(), elements[i], false);

            if (i == elements.length - 1)
            {
                method.invoke(thisObject, values);
            }
            else
            {
                final Object newObject = method.invoke(thisObject, (Object[]) null);

                if (newObject != null)
                {
                	thisObject = newObject;
                }
            }
        }
    }
    
	public static Object getMethodValue(final Object obj, final String fullPath) throws ObjectException
    {
        try
        {
            Method method = null;
            final String[] elements = StringUtility.split(fullPath, PERIOD);
            final Object object = obj;

            if (elements.length == 1)
            {
                final Class superClass = object.getClass();
                method = getMethod(superClass, fullPath, true);

                if (method != null)
                {
                    return method.invoke(object, (Object[]) null);
                }

                return getSuperClassMethod(superClass, object, (Object[]) null, fullPath);
            }
            else
            {
            	return getChildClassMethod(elements, object, (Object[]) null);
            }
        }
        catch (InvocationTargetException exception)
        {
            throw new ObjectException("Method [" + fullPath + "] could not be invoked on Object [" + obj.getClass().getName() + "]", exception);
        }
        catch (IllegalAccessException exception)
        {
            throw new ObjectException("Method [" + fullPath + "] could not be accessed on Object [" + obj.getClass().getName() + "]", exception);
        }
        catch (Exception exception)
        {
            throw new ObjectException("Some error occured when trying to access/invoke Method [" + fullPath + "] on Object [" + obj.getClass().getName() + "]", exception);
        }
    }
    
    private static Object getSuperClassMethod(final Class child, final Object object, final Object[] values, final String fullPath) throws InvocationTargetException, IllegalAccessException
    {
    	Class superClass = child;
    	
    	while ((superClass = superClass.getSuperclass()) != null)
        {
            final Method method = getMethod(superClass, fullPath, true);

            if (method != null)
            {
                return method.invoke(object, values);
            }
        }
    	
    	return null;
    }
    
    private static Object getChildClassMethod(final String[] elements, final Object object, final Object[] values) throws InvocationTargetException, IllegalAccessException
    {
    	Object thisObject = object;
    	
    	for (int i = 0; i < elements.length; i++)
        {
            final Method method = getMethod(thisObject.getClass(), elements[i], true);

            if (i == elements.length - 1)
            {
                return method.invoke(thisObject, (Object[]) null);
            }
            else
            {
                final Object newObject = method.invoke(thisObject, values);

                if (newObject != null)
                {
                	thisObject = newObject;
                }
            }
        }
    	
    	return null;
    }
    
    private static String formatPropertyName(final String propertyName, final boolean getter)
    {
        if (getter)
        {
            if (!propertyName.startsWith(GET) && !StringUtility.isEmpty(propertyName))
            {
                return GET + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
            }
        }
        else
        {
            if (!propertyName.startsWith(SET) && !StringUtility.isEmpty(propertyName))
            {
                return SET + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
            }
        }

        return propertyName;
    }

	private static Method getMethod(final Class CLASS, final String propertyName, final boolean getter)
    {
    	final String methodName = formatPropertyName(propertyName, getter);
        final Method[] methods = CLASS.getDeclaredMethods();

        for (int i = 0; i < methods.length; i++)
        {
            if (methods[i].getName().equals(methodName))
            {
                methods[i].setAccessible(true);
                return methods[i];
            }
        }

        return null;
    }

    public static boolean isNull(final Object object)
    {
        return (object == null);
    }
}
