/**
 * zencore/zen.core.configuration.props/AConfigure.java
 * 
 */
package zen.configuration;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.util.Locale;
import java.util.Properties;
import java.util.Map.Entry;

import zen.file.FileUtility;
import zen.logging.interfaces.ILogging;
import zen.string.StringUtility;

/**
 * Property configuration loader. This class finds and loads the property file specified in
 * IConfiguration.getPropertyFileName(),
 * and returns the configuration loaded with the values from the property file.
 * 
 * @author armin
 */
public abstract class AbstractConfigure implements ILogging
{
	/**
	 * Reads properties into the configuration class.
	 * Only String fields having the same name as a property key are populated.
	 * 
	 * @param iConfiguration
	 *            IConfiguration
	 * @return IConfiguration
	 */
	protected IConfiguration configure(final IConfiguration configuration) throws ConfigurationException
	{
		final Properties properties = this.loadProperties(configuration);
		this.populateConfiguration(configuration, properties);

		return configuration;
	}

	/**
	 * Load the property file defined by IConfiguration.getPropertyFileName()
	 * 
	 * @param configuration
	 * @return Properties
	 */
	private Properties loadProperties(final IConfiguration configuration) throws ConfigurationException
	{
		final Properties properties = new Properties();
		final String configFile = configuration.getPropertyFileName();
		
		try
		{
			final File file = FileUtility.getFile(configFile);
			final FileReader reader = new FileReader(file);
			properties.load(reader);
		}
		catch (UnsupportedEncodingException exception)
		{
			handleException(configFile + " encoding error while loading properties", exception);
		}
		catch (FileNotFoundException exception)
		{
			handleException(configFile + " file could not be found by reader", exception);
		}
		catch (IOException exception)
		{
			handleException(configFile + " IO problem while reading properties", exception);
		}
		catch (Exception exception)
		{
			handleException(configFile + " Unexpected problem while loading properties", exception);
		}

		return properties;
	}
	
	private void handleException(final String message, final Exception exception) throws ConfigurationException
	{
		LOG.fatal(this.getClass(), message, exception);
		throw new ConfigurationException(message, exception);
	}
	
	private void handleWarning(final String message, final Exception exception) 
	{
		LOG.warn(getClass(), message, exception);
	}

	/**
	 * Populate String fields on configuration with properties of the same name.
	 * 
	 * @param IConfiguration
	 *            configuration
	 * @param Properties
	 *            properties
	 */
	private void populateConfiguration(final IConfiguration configuration, final Properties properties)
	{
		for (Entry<Object, Object> prop : properties.entrySet())
		{
			populateConfiguration(prop, configuration);
		}
	}
	
	private void populateConfiguration(final Entry<Object, Object> prop, final IConfiguration configuration) 
	{
		Field field = null;
		
		try
		{
			final StringBuffer propName = new StringBuffer(100);
			final String propKey = (String) prop.getKey();
			
			if (propKey.indexOf('.') >= 0)
			{
				final String[] propParts = propKey.split(".");
				propName.append(propParts[0].toLowerCase(Locale.US));
				
				for (int i = 1; i < propParts.length; i++)
				{
					propName.append(this.camelCase(propParts[i]));
				}
			}
			else
			{
				propName.append(propKey);
			}

			field = configuration.getClass().getDeclaredField(propName.toString());
			field.setAccessible(true);
			
			if (field.getType().getSimpleName().equals("String"))
			{
				field.set(configuration, prop.getValue());
			}
		}
		catch (NoSuchFieldException exception)
		{
			// for getDelcaredField()
			handleWarning((String) prop.getKey() + " does not exist on " + configuration.getClass().getSimpleName() + ".", exception);
			// do not throw anything
		}
		catch (IllegalAccessException exception)
		{
			// for field.set()
			handleWarning((String) prop.getKey() + " of type " + field.getClass().getName() + " is inaccessible.", exception);
			// do not throw anything
		}
	}

	/**
	 * Make the first character of the string uppercase and the remainder lowercase.
	 * 
	 * @param str
	 *            String
	 * @return String
	 */
	private String camelCase(final String string)
	{
		final StringBuffer value = new StringBuffer(10);
		
		if (StringUtility.isEmpty(string))
		{
			value.append("");
		}
		else if (string.length() == 1)
		{
			value.append(string.toUpperCase(Locale.US));
		}
		else
		{
			value.append(string.substring(0, 1).toUpperCase(Locale.US) + string.substring(1).toLowerCase(Locale.US));	
		}

		return value.toString();
	}
}
