package zen.classpath;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.apache.log4j.Logger;

import zen.utility.Utility;

/**
 * TODO: Still need to refactor this code to a better long term solution regarding ClassLoaders
 * 
 * @author David Cano
 *
 */
public final class ClasspathUtility extends Utility
{
	private static final String WEB_INF_CLASSES = "/WEB-INF/classes";
	private static final String WEB_INF_LIB = "/WEB-INF/lib";
	
	private ClasspathUtility()
	{
		super();
		
		//Empty constructor
	}
	
	public static URL getUrlPath(final String path)
    {
		//Had to use getClassLoader to get the correct classpath and not the jar entry location.
		//However, if application is running standalone and not in a server container, the 
		//ClassLoader.getResource will return a null URL. The safety catch is to then switch
		//to the FileUtility.class.getResource. Need a better long term solution! 
		URL url = null;
		
		final ClassLoader loader = Thread.currentThread().getContextClassLoader();
					
		if (loader != null)
		{
			url = loader.getResource(path);
		}
		
		if (url == null)
		{
			url = ClasspathUtility.class.getResource(path);
		}
		
		Logger.getLogger(ClasspathUtility.class.getName()).debug("#### ClasspathUtility URL Path: " + url.toString());
		
		return url;
    }
	
	public static InputStream getInputStream(final String path)
    {
		try
		{
			final URL url = getUrlPath(path);
			
			if (url != null)
			{
				//If the URL actually points to a resource in a jar file, the below has to be done in
				//order to open an InputStream to it, or otherwise you will get a FileNotFoundException
				if (url.getPath().indexOf(".jar!") > -1)
				{
					return getResourceFromJar(path, url);
				}

				return url.openStream();	
			}
		}
		catch (Exception exception)
		{
			Logger.getLogger(ClasspathUtility.class.getName()).warn(exception.toString(), exception);
		}
		
		return null;
    }
	
	private static InputStream getResourceFromJar(final String path, final URL url)
	{
		try
		{
			final String jarFile = url.getPath().substring("file:/".length(), url.getPath().indexOf("!"));
			final File file = new File(URLDecoder.decode(jarFile, "UTF-8"));
			final JarFile jar = new JarFile(file);
			return jar.getInputStream(jar.getEntry(path));
		}
		catch (UnsupportedEncodingException exception)
		{
			Logger.getLogger(ClasspathUtility.class.getName()).warn(exception.toString(), exception);
		}
		catch (IOException exception)
		{
			Logger.getLogger(ClasspathUtility.class.getName()).warn(exception.toString(), exception);
		}

		return null;
	}
	
	public static List<String> getAnnotatedClasses(final Class annotation, final List<String> packages) throws IOException, UnsupportedEncodingException
	{
		final List<String> annotatedClasses = new ArrayList<String>();
		final List<String> classes = getClassesFromClasspath();
		
		for (String className : classes)
		{
			if (isIncluded(className, packages))
			{
				addAnnotatedClass(className, annotation, annotatedClasses);
			}
		}
		
		return annotatedClasses;
	}
	
	private static void addAnnotatedClass(final String className, final Class annotation, final List<String> annotatedClasses)
	{
		try
		{
			final Class classImpl = Class.forName(className);
			
			if (isAnnotationPresent(classImpl, annotation))
			{
				annotatedClasses.add(className);
			}	
		}
		catch (ClassNotFoundException exception)
		{
			//Suppress this exception!
			Logger.getLogger(ClasspathUtility.class.getName()).warn(exception.toString());
		}
	}
	
	public static List<String> getInterfaceClasses(final Class interfaceClass, final List<String> packages) throws IOException, UnsupportedEncodingException
	{
		final List<String> interfaceClasses = new ArrayList<String>();
		final List<String> classes = getClassesFromClasspath();
		
		for (String className : classes)
		{
			if (isIncluded(className, packages))
			{
				addInterfaceClass(className, interfaceClass, interfaceClasses);
			}
		}
		
		return interfaceClasses;
	}
	
	private static void addInterfaceClass(final String className, final Class interfaceClass, final List<String> interfaceClasses)
	{
		try
		{
			final Class implClass = Class.forName(className);
			
			if (hasInterface(implClass, interfaceClass))
			{
				interfaceClasses.add(implClass.getName());	
			}
		}
		catch (ClassNotFoundException exception)
		{
			//Suppress this exception!
			Logger.getLogger(ClasspathUtility.class.getName()).warn(exception.toString());
		}
	}
	
	public static boolean hasInterface(final Class implClass, final Class interfaceClass)
	{
		final Class[] implInterfaces = implClass.getInterfaces();
		
		for (Class implInterface : implInterfaces)
		{
			if (implInterface.getName().equals(interfaceClass.getName()))
			{
				return true;
			}
		}
		
		if (implClass.getSuperclass() != null)
		{
			return hasInterface(implClass.getSuperclass(), interfaceClass);
		}
		
		return false;
	}
	
	public static List<String> getSubClasses(final Class parentClass, final List<String> packages) throws IOException, UnsupportedEncodingException
	{
		final List<String> parentClasses = new ArrayList<String>();
		final List<String> classes = getClassesFromClasspath();
		
		for (String className : classes)
		{
			if (isIncluded(className, packages))
			{
				addSubClass(className, parentClass, parentClasses);
			}
		}
		
		return parentClasses;
	}
	
	private static void addSubClass(final String className, final Class parentClass, final List<String> parentClasses)
	{
		try
		{
			final Class implClass = Class.forName(className);
			
			//Logger.getLogger(ClasspathUtility.class.getName()).info("#### ImplClass: " + className);
			
			if (hasParentClass(implClass, parentClass))
			{
				parentClasses.add(implClass.getName());	
			}
		}
		catch (ClassNotFoundException exception)
		{
			//Suppress this exception!
			Logger.getLogger(ClasspathUtility.class.getName()).warn(exception.toString());
		}
	}
	
	public static boolean hasParentClass(final Class implClass, final Class parentClass)
	{
		Class CLASS = implClass;
		
		while ((CLASS = CLASS.getSuperclass()) != null)
		{
			//Logger.getLogger(ClasspathUtility.class.getName()).info("#### SuperClass: " + CLASS.getName());
			
			if (CLASS.getName().equals(parentClass.getName()))
			{
				return true;
			}
		}
		
		return false;		
	}
	
	public static boolean isAnnotationPresent(final Class implClass, final Class annotation)
	{
		boolean flag = false;
		
		try
		{
			if (implClass.isAnnotationPresent(annotation))
			{
				flag = true;
			}
		}
		catch (Exception exception)
		{
			//Ignoring exceptions coming from Class.forName as exceptions
			//could be thrown for any number of reasons, like for abstracts,
			//singletons, finals, etc.
			Logger.getLogger(ClasspathUtility.class.getName()).info("exception was thrown, but is being ignored");
		}

		return flag;
	}
	
	private static boolean isIncluded(final String className, final List<String> packages)
	{
		if (packages != null && !packages.isEmpty())
		{
			for (String packageName : packages)
			{
				if (className.indexOf(packageName) > -1)
				{
					return true;
				}
			}
		}
		
		return false;
	}
	
	private static List<String> getClassesFromClasspath() throws IOException, UnsupportedEncodingException
	{
		final List<String> classes = new ArrayList<String>();
		final ClassLoader loader = Thread.currentThread().getContextClassLoader();
		final Iterator<File> iterator = getClasspaths(loader.getResources("")).iterator();
		
		while (iterator.hasNext())
		{
			final File directory = (File) iterator.next();
			getClassesFromClasspath(directory, directory, classes);
		}
		
		return classes;
	}
	
	private static void getClassesFromClasspath(final File root, final File directory, final List<String> classes) throws IOException, UnsupportedEncodingException
	{
		final File[] files = directory.listFiles();
		
		for (int i = 0; i < files.length; i++)
		{
			if (files[i].getName().endsWith(".class"))
			{
				final String path = files[i].getPath();
				final String className = path.substring(root.getPath().length() + 1).replace('\\', '/');
				classes.add(getQualifiedClassName(className));
			}
			else if (files[i].getName().endsWith(".jar"))
			{
				classes.addAll(getClassesFromLibrary(files[i].getPath()));
			}
			else if (files[i].isDirectory())
			{
				getClassesFromClasspath(root, files[i], classes);
			}
		}
	}
	
	private static List<String> getClassesFromLibrary(final String library) throws IOException, UnsupportedEncodingException
	{
		final List<String> classes = new ArrayList<String>();
		final JarFile jar = new JarFile(library);
		final Enumeration<JarEntry> enumeration = jar.entries();

		while (enumeration.hasMoreElements())
		{
			final JarEntry entry = (JarEntry) enumeration.nextElement();

			if (entry.getName().endsWith(".class"))
			{
				final String className = getQualifiedClassName(entry.getName());
				classes.add(className);
			}
		}
		
		return classes;
	}
	
	private static List<File> getClasspaths(final Enumeration<URL> enumeration) throws UnsupportedEncodingException
	{
		final List<File> classpaths = new ArrayList<File>();
		
		while (enumeration.hasMoreElements())
		{
			final URL resource = (URL) enumeration.nextElement();
			addClassPath(resource, classpaths);
		}
		
		return classpaths;
	}
	
	private static void addClassPath(final URL resource, final List<File> classpaths) throws UnsupportedEncodingException
	{
		//Must decode the URL path or it will return gibberish the File object does not understand
		final File dir = new File(URLDecoder.decode(resource.getFile(), "UTF-8"));
		
		//The replace call simply makes the code work in both Windows and Unix/Linux
		final String path = dir.getPath().replace('\\', '/');
		
		//If WEB-INF/classes exists, so must WEB-INF/lib, which will not be visible to this class
		//due to the class loading hierarchy used by most web/application servers.
		final int index = path.indexOf(WEB_INF_CLASSES);
		
		if (index > -1)
		{
			final File lib = new File(path.substring(0, index) + WEB_INF_LIB);
			classpaths.add(dir);
			classpaths.add(lib);
		}
		else
		{
			classpaths.add(dir);	
		}
	}

	private static String getQualifiedClassName(final String file)
	{
		return file.substring(0, file.indexOf(".class")).replace('/', '.');
	}
}
