package zen.business;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import zen.business.abstracts.AbstractBusinessService;
import zen.business.interfaces.IBusinessConvention;
import zen.classpath.ClasspathUtility;
import zen.configuration.ConfigurationReader;
import zen.logging.interfaces.ILogging;
import zen.object.ObjectUtility;

public final class BusinessMappings implements ILogging
{
	private static BusinessMappings instance;
	private final transient Map<String, BusinessMapping> mappings = new HashMap<String, BusinessMapping>();

	private BusinessMappings()
	{
		try
		{
			final Configurations configurations = (Configurations) ConfigurationReader.loadConfigurations(Configurations.class);
		    final List<String> interfaceClasses = ClasspathUtility.getInterfaceClasses(IBusinessConvention.class, configurations.getConventions());
		    
		    final List<IBusinessConvention> conventions = new ArrayList<IBusinessConvention>();

            if (interfaceClasses == null || interfaceClasses.isEmpty())
            {
            	LOG.debug(BusinessMappings.class, "No BusinessConventions found...");
            }
            else
            {
            	for (String className : interfaceClasses)
                {
                    LOG.debug(BusinessMappings.class, "BusinessConvention: " + className);
                    final IBusinessConvention convention = (IBusinessConvention) ObjectUtility.createInstance(className);
                    conventions.add(convention);
                }
            }
            
            final List<String> subClasses = ClasspathUtility.getSubClasses(AbstractBusinessService.class, configurations.getServices());
            
            if (subClasses == null || subClasses.isEmpty())
            {
            	LOG.debug(BusinessMappings.class, "No BusinessServices found...");
            }
            else
            {
            	for (String className : subClasses)
                {
                    LOG.debug(BusinessMappings.class, "BusinessService: " + className);

                    try
                    {
                        validate(className, conventions);
                    }
                    catch (BusinessException e)
                    {
                        LOG.error(BusinessMappings.class, e.toString(), e);
                    }
                }
            }
		}
		catch (Exception ex)
		{
			LOG.error(BusinessMappings.class, ex.toString(), ex);
		}
	}

	private void validate(final String serviceName, final List<IBusinessConvention> conventions) throws BusinessException
	{
	    final ConventionMemento memento = new ConventionMemento(serviceName);
	    
	    for (IBusinessConvention convention : conventions)
	    {
	        convention.validate(memento);
	    }
		
		if (memento.getMessages().isEmpty())
		{
			final BusinessMapping mapping = new BusinessMapping();
			mapping.setService(memento.getServiceName());
			mapping.setResponse(memento.getResponseName());
			mappings.put(memento.getRequestName(), mapping);
		}
		else
		{
			throw new BusinessException(generateMessage(memento.getMessages()));	
		}
	}
	
	private String generateMessage(final List<String> messages)
	{
	    final StringBuffer message = new StringBuffer();
	    
	    for (String line : messages)
	    {
	        message.append("\n\t");
	        message.append(line);
	    }
	    
	    return message.toString();
	}

	public static BusinessMappings getInstance()
	{
		if (instance == null)
		{
			instance = new BusinessMappings();
		}

		return instance;
	}

	public BusinessMapping getMapping(final String requestName) throws BusinessException
	{
		if (mappings.containsKey(requestName))
		{
			return (BusinessMapping) mappings.get(requestName);
		}

		throw new BusinessException("Mapping does not exist for " + requestName);
	}
}
