package zen.business;

import zen.business.abstracts.AbstractBusinessRequest;
import zen.business.abstracts.AbstractBusinessResponse;
import zen.business.abstracts.AbstractBusinessService;
import zen.logging.interfaces.ILogging;

public class BusinessController implements ILogging
{
	public final AbstractBusinessResponse consume(final AbstractBusinessRequest request) throws BusinessException
	{
		if (request == null)
		{
			throw new BusinessException("Request was null!");
		}
		else
		{
			final String requestName = request.getClass().getName();
			final BusinessMapping mapping = BusinessMappings.getInstance().getMapping(requestName);

			if (mapping == null)
			{
				throw new BusinessException("Mapping not found for " + requestName);
			}
			else
			{
				return callService(request, mapping);
			}
		}
	}

	private final AbstractBusinessResponse callService(final AbstractBusinessRequest request, final BusinessMapping mapping) throws BusinessException
	{
		AbstractBusinessResponse response = null;
		AbstractBusinessService service = null;

		try
		{
			response = (AbstractBusinessResponse) Class.forName(mapping.getResponse()).newInstance();
			service = (AbstractBusinessService) Class.forName(mapping.getService()).newInstance();

			LOG.info(getClass(), getCallDetails(service, request, response));

			service.consume(request, response);
		}
		catch(IllegalAccessException exception)
		{
			handleException(exception, service, response);
		}
		catch (InstantiationException exception)
		{
			handleException(exception, service, response);
		}
		catch (ClassNotFoundException exception)
		{
			handleException(exception, service, response);
		}
		catch (Exception exception)
		{
			handleException(exception, service, response);
		}

		return response;
	}
	
	private final void handleException(final Exception exception, final AbstractBusinessService service, final AbstractBusinessResponse response) throws BusinessException
	{
		if (response == null)
		{
			LOG.error(getClass(), exception.toString(), exception);
			throw new BusinessException("Error calling service: " + exception.toString(), exception);
		}
		else
		{
			LOG.error(getClass(), "Service [" + service.getClass().getName() + "] Error [" + exception.toString() + "]", exception);
			response.addMessage(new BusinessError(exception.toString()));	
		}
	}
	
	private final String getCallDetails(final AbstractBusinessService service, final AbstractBusinessRequest request, final AbstractBusinessResponse response)
	{
		final StringBuffer statement = new StringBuffer(100);
		statement.append("Service Call To ");
		statement.append(service.getClass().getName());
		statement.append(".consume(");
		statement.append(request.getClass().getName());
		statement.append(", ");
		statement.append(response.getClass().getName());
		statement.append(')');
		return statement.toString();
	}
}
