/* 
 * Copyright (c) 2005-2011, Fraunhofer-Gesellschaft
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * (1) Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the disclaimer at the end.
 *     Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * 
 * (2) Neither the name of Fraunhofer nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 */
package org.ogf.graap.wsag.api.types;

import org.apache.xmlbeans.XmlObject;
import org.ogf.graap.wsag.api.AgreementFactory;
import org.ogf.schemas.graap.wsAgreement.AgreementFactoryPropertiesType;
import org.ogf.schemas.graap.wsAgreement.AgreementTemplateType;

/**
 * AbstractAgreementFactoryType
 * 
 * @author Oliver Waeldrich
 */
public abstract class AbstractAgreementFactoryType extends WSAGXmlType implements AgreementFactory
{

    private AgreementFactoryPropertiesType delegateProps = null;

    /**
     * Default constructor
     */
    public AbstractAgreementFactoryType()
    {
        initialize();
    }

    /**
     * initializes the agreement factory properties of this instance
     */
    protected void initialize()
    {
        delegateProps = AgreementFactoryPropertiesType.Factory.newInstance();
    }

    /**
     * validates the agreement factory properties of this instance
     * 
     * {@inheritDoc}
     */
    public boolean validate()
    {
        return validate( delegateProps );
    }

    /**
     * Adds a new template to the agreement factory properties of this instance
     * 
     * @return the new template
     */
    public AgreementTemplateType addNewTemplate()
    {
        return delegateProps.addNewTemplate();
    }

    /**
     * @return the templates supported by this factory instance
     */
    public AgreementTemplateType[] getTemplates()
    {
        return delegateProps.getTemplateArray();
    }

    /**
     * @param index
     *            specifies the template to return
     * @return the template specified by the index
     */
    public AgreementTemplateType getTemplateArray( int index )
    {
        return delegateProps.getTemplateArray( index );
    }

    /**
     * Sets the agreement templates for this factory
     * 
     * @param templateArray
     *            the template array to set
     */
    public void setTemplateArray( AgreementTemplateType[] templateArray )
    {
        delegateProps.setTemplateArray( templateArray );
    }

    /**
     * Updates the template at the position specified by the index parameter.
     * 
     * @param index
     *            the index of the template to set
     * @param template
     *            the template top set
     */
    public void setTemplateArray( int index, AgreementTemplateType template )
    {
        delegateProps.setTemplateArray( index, template );
    }

    /**
     * @return the number of templates exposed by this factory instance
     */
    public int sizeOfTemplateArray()
    {
        return delegateProps.sizeOfTemplateArray();
    }

    /**
     * @return the XML representation of the agreement factory resource properties
     */
    public XmlObject getXMLObject()
    {
        return delegateProps;
    }
}
