/* 
 * Copyright (c) 2007, Fraunhofer-Gesellschaft
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * (1) Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the disclaimer at the end.
 *     Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * 
 * (2) Neither the name of Fraunhofer nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 */
package org.ogf.graap.wsag.api.client;

import org.ogf.graap.wsag.api.exceptions.NegotiationException;
import org.ogf.graap.wsag.api.exceptions.ResourceUnavailableException;
import org.ogf.graap.wsag.api.exceptions.ResourceUnknownException;
import org.ogf.schemas.graap.wsAgreement.AgreementTemplateType;
import org.ogf.schemas.graap.wsAgreement.negotiation.NegotiationContextType;
import org.ogf.schemas.graap.wsAgreement.negotiation.NegotiationOfferType;

/**
 * NegotiationService
 * 
 * @author hrasheed
 */
public interface NegotiationService
{

    /**
     * Returns the context of this negotiation process.
     * 
     * @return the negotiation context
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    NegotiationContextType getNegotiationContext()
        throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * A negotiation instance supports zero or more templates that can be used to create negotiation offers.
     * In case of a negotiation process, these templates are a subset of the templates offered by an agreement
     * factory. Only these templates are returned for which a proper negotiation strategy is implemented. in
     * case of agreement re negotiation, a negotiation instance may return one ore more templates to bootstrap
     * the negotiation process.
     * 
     * @return the negotiable templates
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    AgreementTemplateType[] getNegotiableTemplates()
        throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Returns a history of exchanged offers in this negotiation process.
     * 
     * @return the offers exchanged in this negotiation
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    NegotiationOfferType[] getNegotiationOffers()
        throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Negotiates a set of agreement offers.
     * 
     * @param offers
     *            the offers to negotiate
     * @return a set of counter offers
     * @throws NegotiationException
     *             an error occured processing the negotiation request
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    NegotiationOfferType[] negotiate( NegotiationOfferType[] offers )
        throws NegotiationException, ResourceUnknownException, ResourceUnavailableException;

    /**
     * Advertises a set of negotiation offers.
     * 
     * @param offers
     *            the offers to advertise
     * @throws NegotiationException
     *             an error occured processing the negotiation request
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    void advertise( NegotiationOfferType[] offers )
        throws NegotiationException, ResourceUnknownException, ResourceUnavailableException;

    /**
     * Terminates a negotiation process.
     * 
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    void terminate() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * @return the web service client for this service.
     */
    WsClient getWebServiceClient();

    /**
     * Destroys this web service object.
     * 
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    void destroy() throws ResourceUnknownException, ResourceUnavailableException;

}
