/* 
 * Copyright (c) 2007, Fraunhofer-Gesellschaft
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * (1) Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the disclaimer at the end.
 *     Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * 
 * (2) Neither the name of Fraunhofer nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 */
package org.ogf.graap.wsag.api.client;

import org.ogf.graap.wsag.api.exceptions.ResourceUnavailableException;
import org.ogf.graap.wsag.api.exceptions.ResourceUnknownException;
import org.ogf.schemas.graap.wsAgreement.AgreementContextType;
import org.ogf.schemas.graap.wsAgreement.AgreementStateType;
import org.ogf.schemas.graap.wsAgreement.GuaranteeTermStateType;
import org.ogf.schemas.graap.wsAgreement.ServiceTermStateType;
import org.ogf.schemas.graap.wsAgreement.TermTreeType;
import org.ogf.schemas.graap.wsAgreement.TerminateInputType;

/**
 * AgreementService
 * 
 * @author Oliver Waeldrich
 */
public interface AgreementService
{

    /**
     * Returns the web-service client used for communication with the SLA management server.
     * 
     * @return the web service client for this service.
     */
    WsClient getWebServiceClient();

    //
    // from the ArgreementPortType
    //

    /**
     * Returns the name of the agreement instance.
     * 
     * @return the agreement name
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    String getName() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Returns the id of the agreement instance.
     * 
     * @return the agreement id
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    String getAgreementId() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Returns the context of the agreement instance.
     * 
     * @return the agreement context
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    AgreementContextType getContext() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Returns the terms of the agreement instance.
     * 
     * @return the agreement terms
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    TermTreeType getTerms() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Terminates the agreement instance.
     * 
     * @param reason
     *            a domain-specific termination reason
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    void terminate( TerminateInputType reason ) throws ResourceUnknownException, ResourceUnavailableException;

    //
    // from the ArgreementStatePortType
    //

    /**
     * Returns the state of the agreement instance.
     * 
     * @return the agreement state
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    AgreementStateType getState() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Returns the state for the individual guarantee terms of the agreement instance.
     * 
     * @return the guarantee term states
     * 
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    GuaranteeTermStateType[] getGuaranteeTermStates()
        throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Returns the state for the individual service terms of the agreement instance.
     * 
     * @return the service term states
     * 
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    ServiceTermStateType[] getServiceTermStates()
        throws ResourceUnknownException, ResourceUnavailableException;

    //
    // from WS resource specification
    //

    /**
     * Destroys agreement instance.
     * 
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    void destroy() throws ResourceUnknownException, ResourceUnavailableException;

}
