/* 
 * Copyright (c) 2007, Fraunhofer-Gesellschaft
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * (1) Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the disclaimer at the end.
 *     Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * 
 * (2) Neither the name of Fraunhofer nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 */
package org.ogf.graap.wsag.api.client;

import org.ogf.graap.wsag.api.AgreementOffer;
import org.ogf.graap.wsag.api.exceptions.AgreementFactoryException;
import org.ogf.graap.wsag.api.exceptions.NegotiationFactoryException;
import org.ogf.graap.wsag.api.exceptions.ResourceUnavailableException;
import org.ogf.graap.wsag.api.exceptions.ResourceUnknownException;
import org.ogf.schemas.graap.wsAgreement.AgreementTemplateType;
import org.ogf.schemas.graap.wsAgreement.negotiation.NegotiationContextType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

/**
 * {@link AgreementFactoryService} The contract of a WSAG4J agreement factory service.
 * 
 * @author Oliver Waeldrich
 */
public interface AgreementFactoryService
    extends WSDMResource
{

    /**
     * Returns the web-service client used for communicating with the SLA management server.
     * 
     * @return the web service client for this service.
     */
    WsClient getWebServiceClient();

    /**
     * Retrieves the templates for supported SLAs from the SLA management system.
     * 
     * @return all templates exposed by this agreement factory instance
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    AgreementTemplateType[] getTemplates() throws ResourceUnknownException, ResourceUnavailableException;

    /**
     * Creates a new agreement instance with the SLA management server.
     * 
     * @param offer
     *            the agreement offer
     * @return a client for the created agreement
     * @throws AgreementFactoryException
     *             indicates that the agreement was rejected by the factory
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    AgreementClient createAgreement( AgreementOffer offer )
        throws AgreementFactoryException, ResourceUnknownException, ResourceUnavailableException;

    /**
     * Creates a pending agreement instance. The decision whether to accept or reject the agreement is
     * deferred.
     * 
     * @param offer
     *            The agreement offer passed from the agreement initiator.
     * @return An {@link AgreementClient} for the new created agreement.
     * @throws AgreementFactoryException
     *             An error occurred during the the agreement creation. The agreement was rejected.
     * @throws ResourceUnknownException
     *             No pending agreement factory exists at the specified endpoint.
     * @throws ResourceUnavailableException
     *             The agreement factory at the specified endpoint is not accessible.
     */
    AgreementClient createPendingAgreement( AgreementOffer offer )
        throws AgreementFactoryException, ResourceUnknownException, ResourceUnavailableException;

    /**
     * Creates a pending agreement instance. The decision whether to accept or reject the agreement is
     * deferred. Once the decision has been made the client is notified via the given notification endpoint.
     * 
     * @param offer
     *            The agreement offer passed from the agreement initiator.
     * @param acceptanceEPR
     *            An endpoint to an agreement acceptance instance.
     * @return An {@link AgreementClient} for the new created agreement.
     * @throws AgreementFactoryException
     *             An error occurred during the the agreement creation. The agreement was rejected.
     * @throws ResourceUnknownException
     *             No pending agreement factory exists at the specified endpoint.
     * @throws ResourceUnavailableException
     *             The agreement factory at the specified endpoint is not accessible.
     */
    AgreementClient createPendingAgreement( AgreementOffer offer, EndpointReferenceType acceptanceEPR )
        throws AgreementFactoryException, ResourceUnknownException, ResourceUnavailableException;

    /**
     * Initiates a new negotiation process.
     * 
     * @param context
     *            the context for the negotiation to initialize
     * @return the new initiated negotiation instance
     * @throws NegotiationFactoryException
     *             indicates that the creation of a negotiation process was rejected
     * @throws ResourceUnknownException
     *             the remote resource is unknown
     * @throws ResourceUnavailableException
     *             the remote resource is unavailable
     */
    NegotiationClient initiateNegotiation( NegotiationContextType context )
        throws NegotiationFactoryException, ResourceUnknownException, ResourceUnavailableException;
}
