/* 
 * Copyright (c) 2007, Fraunhofer-Gesellschaft
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * (1) Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the disclaimer at the end.
 *     Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * 
 * (2) Neither the name of Fraunhofer nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 */
package org.ogf.graap.wsag.api;

import org.ogf.graap.wsag.api.types.AbstractAgreementType;
import org.ogf.schemas.graap.wsAgreement.AgreementContextType;
import org.ogf.schemas.graap.wsAgreement.AgreementStateType;
import org.ogf.schemas.graap.wsAgreement.GuaranteeTermStateType;
import org.ogf.schemas.graap.wsAgreement.ServiceTermStateType;
import org.ogf.schemas.graap.wsAgreement.TermTreeType;
import org.ogf.schemas.graap.wsAgreement.TerminateInputType;

/**
 * Agreement interface used by the server module. It provides the required methods to access the properties of
 * the agreement and to terminate the agreement instance. Moreover, it provides access to the domains-specific
 * agreement implementation via the {@link #getAgreementInstance()} method.
 * 
 * @see AgreementFactory
 * @author Oliver Waeldrich
 */
public interface Agreement
{

    // definitions from the ArgreementPortType

    /**
     * Returns the agreement name.
     * 
     * @return the agreement name
     */
    String getName();

    /**
     * Returns the agreement id.
     * 
     * @return the agreement id
     */
    String getAgreementId();

    /**
     * Returns a domain-specific agreement implementation. The {@link AbstractAgreementType} implements the
     * domain-specific termination method and stored the agreement properties. The
     * {@link AbstractAgreementType#getXMLObject()} returns the actual properties of an agreement
     * implementation. These properties must be valid at all time, i.e. if an agreement implementation
     * overrides the {@link #getState()} method the implementation must make sure that the agreement
     * properties are updated with the returned state.
     * 
     * @return the agreement implementation
     */
    AbstractAgreementType getAgreementInstance();

    /**
     * @return the agreement context
     */
    AgreementContextType getContext();

    /**
     * @return the agreement terms
     */
    TermTreeType getTerms();

    /**
     * Terminates an agreement.
     * 
     * @param reason
     *            termination reason
     */
    void terminate( TerminateInputType reason );

    // definitions from the ArgreementStatePortType

    /**
     * @return the agreement state
     */
    AgreementStateType getState();

    /**
     * @return the agreement guarantee term states
     */
    GuaranteeTermStateType[] getGuaranteeTermStates();

    /**
     * @return the agreement service term states
     */
    ServiceTermStateType[] getServiceTermStates();

}
