package net.sf.testium.selenium;

import org.openqa.selenium.By;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.pagefactory.ElementLocator;

import java.lang.reflect.Field;
import java.util.List;

/**
 * The Smart element locator, which will lazily locate an element or an element
 * list on a page. This class is designed for use with the
 * {@link org.openqa.selenium.support.PageFactory} and understands the
 * annotations {@link org.openqa.selenium.support.FindBy},
 * {@link org.openqa.selenium.support.CacheLookup}, and
 * {@link net.sf.testium.selenium.TestiumVar}
 */
public class SmartElementLocator implements ElementLocator {

	private final WebDriverInterface iface;
	private final boolean shouldCache;
	private final By by;
	private final String testiumName;

	private SmartWebElement cachedElement;
	private SmartWebElementList cachedElementList;

	/**
	 * Creates a new element locator.
	 * 
	 * @param searchContext
	 *            The context to use when finding the element
	 * @param field
	 *            The field on the Page Object that will hold the located value
	 */
	public SmartElementLocator(
			WebDriverInterface iface, Field field) {
		this.iface = iface;
		TestiumAnnotations annotations = new TestiumAnnotations(field);
		shouldCache = annotations.isLookupCached();
		by = annotations.buildBy();
		testiumName = annotations.getTestiumName();
	}

	/**
	 * Find the element.
	 */
	public SmartWebElement findElement() {
		if (cachedElement != null && shouldCache) {
			return cachedElement;
		}

		WebDriver driver = iface.getDriver();
		WebElement element = driver.findElement(by);
		SmartWebElement smartElm = new SimplePageElement(by, iface, element);
		if (shouldCache) {
			cachedElement = smartElm;
		}

		// It could be that the proxy is stored. In any case we republish the element.
		if (driver instanceof FieldPublisher && testiumName != "") {
			FieldPublisher elementMap = (FieldPublisher) driver;

			elementMap.addElement(testiumName, smartElm);
		}

		return smartElm;
	}

	/**
	 * Find the element list.
	 */
	public SmartWebElementList findElements() {
		if (cachedElementList != null && shouldCache) {
			return cachedElementList;
		}

		WebDriver driver = iface.getDriver();
		List<WebElement> webElements = driver.findElements(by);
		SmartWebElementList elements = new SimpleElementList( by, iface, webElements );
		if (shouldCache) {
			cachedElementList = elements;
		}

		return elements;
	}

	public String getTestiumName() {
		return testiumName;
	}

}
