/**
 * 
 */
package net.sf.testium.executor;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Hashtable;

import net.sf.testium.configuration.PerlConfiguration;

import net.sf.testium.executor.TestCaseExecutor;
import org.testtoolinterfaces.testresult.TestCaseResult;
import org.testtoolinterfaces.testresult.TestCaseResultLink;
import org.testtoolinterfaces.testresult.TestResult.VERDICT;
import org.testtoolinterfaces.testresultinterface.TestCaseResultReader;
import org.testtoolinterfaces.testresultinterface.TestCaseResultWriter;
import org.testtoolinterfaces.testsuite.LooseTestInterfaceList;
import org.testtoolinterfaces.testsuite.TestCase;
import org.testtoolinterfaces.testsuite.TestCaseImpl;
import org.testtoolinterfaces.testsuite.TestCaseLink;
import org.testtoolinterfaces.testsuite.TestInterfaceList;
import org.testtoolinterfaces.testsuite.TestStepSequence;
import org.testtoolinterfaces.utils.RunTimeData;
import org.testtoolinterfaces.utils.Trace;

/**
 * @author Arjan Kranenburg
 *
 */
public class TestCasePerlExecutor implements TestCaseExecutor
{
	public static final String TYPE = "perl";
	
	private PerlConfiguration myConfiguration;
	private TestCaseResultWriter myTestCaseResultWriter;
	private TestCaseResultReader myTestCaseResultReader;

	public TestCasePerlExecutor( PerlConfiguration aConfig,
	                             TestCaseResultWriter aTestCaseResultWriter )
	{
		Trace.println(Trace.CONSTRUCTOR);

		myConfiguration = aConfig;
		myTestCaseResultWriter = aTestCaseResultWriter;
		TestInterfaceList interfaceList = new LooseTestInterfaceList();
		myTestCaseResultReader = new TestCaseResultReader( interfaceList );
		
	}

    /**
     * Executes the Test Case as a perl script on the Operating System
     * 
	 * @param aTestCaseLink the link to the perl TestCase
	 * @param aLogDir the baseDir for log-files
	 * @param anRTData the collection of runtime variables
     */
    public TestCaseResultLink execute( TestCaseLink aTestCaseLink,
                                       File aLogDir,
                                       RunTimeData anRTData)
    {
    	String tcId = aTestCaseLink.getId();
		Trace.println(Trace.EXEC, "execute( "
						+ tcId + ", "
			            + aLogDir.getPath() + ", "
			            + anRTData.size() + " Variables )", true );

		if ( !aLogDir.isDirectory() )
		{
			FileNotFoundException exc = new FileNotFoundException("Directory does not exist: " + aLogDir.getPath());
			throw new IOError( exc );
		}

		File executable = aTestCaseLink.getLink();

		File caseLogDir = new File(aLogDir, tcId);
		caseLogDir.mkdir();
		File resultFile = new File( caseLogDir, tcId + ".xml" );
		File origResultFile = new File( caseLogDir, tcId + "_orig.xml" );

		TestCaseResult result;
		File logFile = new File( caseLogDir, tcId + "_run.log" );
		try
		{
			VERDICT retCodeVerdict = PerlScript.execute(executable, resultFile, logFile, myConfiguration);
			copy( resultFile, origResultFile );
			
			result = readResultFile( resultFile );
			result.setResult(retCodeVerdict);
    		myTestCaseResultWriter.write( result, resultFile );
		}
		catch (FileNotFoundException e)
		{
			String description = getDescription( executable );
			ArrayList<String> requirements = getRequirements( executable );

			TestCase pseudoTestCase = new TestCaseImpl( tcId,
				                                        description,
				                                        0,
				                                        requirements,
				                                        new TestStepSequence(),
				                                        new TestStepSequence(),
				                                        new TestStepSequence(),
					                                    new Hashtable<String, String>(),
				                                        new Hashtable<String, String>());

			result = new TestCaseResult( pseudoTestCase );
			resultFile = new File( caseLogDir, tcId + "_error.xml" );
    		myTestCaseResultWriter.write( result, resultFile );

    		result.addComment( "Execution Failed: " + e.getLocalizedMessage() );
    		StackTraceElement[] elms = e.getStackTrace();
    		for (int key = 0; key < elms.length; key++)
        	{
        		result.addComment( elms[key].toString() );
        	}
			result.setResult( VERDICT.ERROR );
		}
		result.addTestLog("log", logFile.getPath());
    	
		return new TestCaseResultLink( aTestCaseLink,
		                               result.getResult(),
		                               resultFile );
    }

	public String getType()
	{
		return TYPE;
	}
	
	private String getDescription( File aPerlScript )
	{
		// TODO try to get a description from the perl script
		return "";
	}
	
	private ArrayList<String> getRequirements( File aPerlScript )
	{
		// TODO try to get a requirements from the perl script
		return new ArrayList<String>();
	}
	
	private TestCaseResult readResultFile( File aResultFile )
						   throws FileNotFoundException
	{
		try
		{
			return myTestCaseResultReader.readTcResultFile(aResultFile);
		}
		catch( IOError anIoError )
		{
			String msg = anIoError.getMessage();
    		StackTraceElement[] elms = anIoError.getStackTrace();
    		for (int key = 0; key < elms.length; key++)
        	{
    			msg += "\n" + elms[key].toString();
        	}
			throw new FileNotFoundException( msg );
		}
	}
	
	// Copies src file to dst file.
	// If the dst file does not exist, it is created
	private void copy(File src, File dst)
	{
		try
		{
		    InputStream in = new FileInputStream(src);
		    OutputStream out = new FileOutputStream(dst);

		    // Transfer bytes from in to out
		    byte[] buf = new byte[1024];
		    int len;
		    while ((len = in.read(buf)) > 0)
		    {
		        out.write(buf, 0, len);
		    }
		    in.close();
		    out.close();
		}
		catch (IOException e)
		{
			Trace.print( Trace.EXEC_PLUS, e );
		}
	}
}
