package net.sf.testium.executor;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;

import net.sf.testium.configuration.PerlConfiguration;
import org.testtoolinterfaces.testresult.TestResult;
import org.testtoolinterfaces.testsuite.Parameter;
import org.testtoolinterfaces.testsuite.ParameterImpl;
import org.testtoolinterfaces.testsuite.ParameterArrayList;
import org.testtoolinterfaces.utils.StreamGobbler;
import org.testtoolinterfaces.utils.Trace;



public class PerlScript
{
    /**
     * See http://www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html
     * 
	 * @param aScript the script to execute, without parameters
	 * @param aResultFile the TTI Test Case Result File
	 * @param aRunLog the file to store the output
	 * @param aConfiguration the Perl settings to use
	 * 
     * @throws FileNotFoundException when the script or run-log does not
     *         exist or are directories
     */
    public static TestResult.VERDICT execute( File aScript, File aResultFile, File aRunLog, PerlConfiguration aConfiguration ) throws FileNotFoundException
    {
    	return execute( aScript, new ParameterArrayList(), aResultFile, aRunLog, aConfiguration );
    }

    /**
     * See http://www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html
     * 
	 * @param aScript the script to execute, without parameters
	 * @param aParameters a table of parameters to use
	 * @param aResultFile the TTI Test Case Result File
	 * @param aRunLog the file to store the output
	 * @param aConfiguration the Perl settings to use
	 * 
     * @throws FileNotFoundException when the base log dir does not
     *         exist or is not a directory
     */
    public static TestResult.VERDICT execute( File aScript,
                                              ParameterArrayList aParameters,
                                              File aResultFile,
                                              File aRunLog,
                                              PerlConfiguration aConfiguration ) throws FileNotFoundException
    {
		if ( aScript==null )
		{
			throw new NullPointerException("Script cannot be null");
		}
		if ( aScript.isDirectory() )
		{
			throw new FileNotFoundException("Script cannot be a directory: " + aScript.getPath());
		}

		if ( aResultFile==null )
		{
			throw new NullPointerException("Result File cannot be null");
		}
		if ( aResultFile.isDirectory() )
		{
			throw new FileNotFoundException("Result File cannot be a Directory: " + aResultFile.getPath());
		}

		if ( aRunLog==null )
		{
			throw new NullPointerException("Run log cannot be null");
		}
		if ( aRunLog.isDirectory() )
		{
			throw new FileNotFoundException("Run log cannot be a Directory: " + aRunLog.getAbsolutePath());
		}

		Trace.println(Trace.EXEC_PLUS, "execute( "
				+ aScript.getPath() + ", "
				+ aParameters.size() + " Parameters, "
				+ aResultFile.getPath() + ", "
	            + aRunLog.getPath() + ", " 
	            + "PerlConfiguration )", true );

        FileOutputStream runLog = new FileOutputStream(aRunLog.getAbsolutePath());
    	if ( !aScript.canExecute() )
    	{
    		PrintWriter pw = new PrintWriter(runLog);
    		pw.println("Cannot execute file:");
    		pw.println(aScript.getAbsolutePath());
            pw.flush();

			throw new FileNotFoundException("Cannot execute file: " + aScript.getName());
    	}

    	String baseLogName = aRunLog.getName().substring(0, aRunLog.getName().lastIndexOf('.'));
    	
    	// On windows, this could be cmd /c perl -f <script> or so. Can we detect the OS?
    	String commandString = aScript.getAbsolutePath();
    	ArrayList<Parameter> params = aParameters.sort();
    	for(int i=0; i<params.size(); i++)
    	{
    		if ( ParameterImpl.class.isInstance( params.get(i) ) )
    		{
    			ParameterImpl param = (ParameterImpl) params.get(i);
        		commandString += " " + param.getName() + " " + param.getValue().toString();
    		}
    	}

    	Trace.println(Trace.EXEC_PLUS, "Executing " + commandString);
    	Trace.println(Trace.EXEC_PLUS, "Writing result to " + aResultFile.getAbsolutePath());
    	Trace.println(Trace.EXEC_PLUS, "Writing log to " + aRunLog.getAbsolutePath());
    	
    	commandString += " " + aConfiguration.getLogFileOption();
    	commandString += " " + aResultFile.getAbsolutePath();
    	if ( aConfiguration.getPassSettings() )
    	{
    		commandString += " " + aConfiguration.getSettingsOption();
    		commandString += " " + aConfiguration.getSettingsFile().getAbsolutePath();
    	}
    	
		File commandLogFile = new File(aRunLog.getParent(), baseLogName + "_command.log");
		BufferedWriter commandLog;
		try
		{
			commandLog = new BufferedWriter(new FileWriter(commandLogFile));
			commandLog.write( commandString + "\n" );
			commandLog.close();
		}
		catch (IOException e)
		{
			// Ignored. Then we just don't log the command
		}

//		File errorLogFile = new File(aRunLog.getParent(), baseLogName + "_error.log");
//        FileOutputStream errorLog = new FileOutputStream(errorLogFile.getAbsolutePath());
        try
        {
            Runtime rt = Runtime.getRuntime();
            Process proc = rt.exec(commandString);
            // TODO Process proc = new ProcessBuilder(commandString).start();

            // any error message?
            StreamGobbler errorGobbler = new 
//            StreamGobbler(proc.getErrorStream(), errorLog);            
            StreamGobbler(proc.getErrorStream(), runLog);            
            
            // any output?
            StreamGobbler outputGobbler = new 
                StreamGobbler(proc.getInputStream(), runLog);
                
            // kick them off
            errorGobbler.start();
            outputGobbler.start();
            
            // Wait for threads to finish
            int exitVal = proc.waitFor();
    		try
    		{
    			Thread.sleep( 20 );
    		}
    		catch (InterruptedException e)
    		{
    			throw new Error( e );
    		}
            runLog.flush();
//            errorLog.flush();
            runLog.close();
//            errorLog.close();


    		Trace.println(Trace.EXEC_PLUS, "Exit value is " + exitVal);
            if ( exitVal > 0 )
            {
                return TestResult.FAILED;
            }
            else
            {
                return TestResult.PASSED;
            }
        }
        catch (Throwable t)
        {
        	Long logSize = new Long(0);
            if ( aRunLog.exists() && aRunLog.isFile() )
            {
            	logSize = aRunLog.length();
            }

        	if ( logSize.intValue() == 0 )
        	{
        		PrintWriter pw = new PrintWriter(runLog);
        		pw.println("Error while executing perl script:");
        		pw.println(t.getMessage());
                pw.flush();
        	}
			throw new FileNotFoundException("Error while executing perl script: " + t.getMessage());
        }
    }
}
