//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc.xml;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;

/**
 * @author Norbert Sándor
 */
public class XsltUtils
{
    private XsltUtils()
    {
    }

    public static void xsltTransform(Source xmlSource, Result output, Reader xsltReader, Map<String, ?> parameters)
            throws XsltException
    {
        try
        {
            TransformerFactory factory = TransformerFactory.newInstance();
            Templates template = factory.newTemplates(new StreamSource(xsltReader));

            Transformer transformer = template.newTransformer();

            if (parameters != null)
            {
                for (Map.Entry<String, ?> entry : parameters.entrySet())
                {
                    transformer.setParameter(entry.getKey(), entry.getValue());
                }
            }

            transformer.transform(xmlSource, output);
        }
        catch (TransformerConfigurationException e)
        {
            throw new XsltException(e);
        }
        catch (TransformerException e)
        {
            throw new XsltException(e);
        }
    }

    public static void xsltTransform(Source xmlSource, Result output, Reader xsltReader) throws XsltException
    {
        xsltTransform(xmlSource, output, xsltReader, null);
    }

    public static String xsltTransform(String xml, Reader xsltReader, Map<String, ?> parameters)
    {
        Source xmlSource = new StreamSource(new StringReader(xml));

        StringWriter writer = new StringWriter();
        Result output = new StreamResult(writer);

        xsltTransform(xmlSource, output, xsltReader, parameters);

        return writer.toString();
    }

    public static String xsltTransform(String xml, InputStream xsltStream, String... parameters)
    {
        if (parameters.length % 2 != 0)
        {
            throw new IllegalArgumentException("Even number of parameters expected (name-value pairs)");
        }

        Map<String, String> parameterMap = new HashMap<String, String>();
        for (int i = 0; i < parameters.length; i += 2)
        {
            parameterMap.put(parameters[i], parameters[i + 1]);
        }

        return xsltTransform(xml, xsltStream, parameterMap);
    }

    public static String xsltTransform(String xml, InputStream xsltStream, Map<String, ?> parameters)
    {
        return xsltTransform(xml, new InputStreamReader(xsltStream), parameters);
    }

    public static String xsltTransform(String xml, InputStream xsltStream)
    {
        return xsltTransform(xml, xsltStream, (Map<String, ?>) null);
    }

    public static void writeXmlDocument(Source xmlSource, Result output)
    {
        try
        {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.transform(xmlSource, output);
        }
        catch (TransformerConfigurationException e)
        {
            throw new XsltException(e);
        }
        catch (TransformerException e)
        {
            throw new XsltException(e);
        }
    }

    public static void writeXmlDocument(Document document, Writer writer)
    {
        writeXmlDocument(new DOMSource(document), new StreamResult(writer));
    }
}
