//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc.service.xdoc;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.hivemind.ClassResolver;
import org.apache.hivemind.Resource;
import org.apache.hivemind.util.ClasspathResource;

import com.erinors.tapestry.tapdoc.model.Component;
import com.erinors.tapestry.tapdoc.model.Library;
import com.erinors.tapestry.tapdoc.service.DocumentGenerator;
import com.erinors.tapestry.tapdoc.service.FileNameGenerator;
import com.erinors.tapestry.tapdoc.util.ZipUtils;
import com.erinors.tapestry.tapdoc.xml.XsltUtils;

/**
 * @author Norbert Sándor
 */
public class XdocDocumentGenerator implements DocumentGenerator
{
    public XdocDocumentGenerator(ClassResolver classResolver, FileNameGenerator fileNameGenerator)
    {
        this.classResolver = classResolver;
        this.fileNameGenerator = fileNameGenerator;
    }

    private final ClassResolver classResolver;

    private final FileNameGenerator fileNameGenerator;

    public void generate(FileObject outputDirectory, FileObject generatedOutputDirectory, List<Library> libraryModels,
            String tapdocXml) throws FileSystemException
    {
        //
        // Make output directory
        //

        if (!generatedOutputDirectory.exists())
        {
            generatedOutputDirectory.createFolder();
        }

        if (outputDirectory.exists())
        {
            outputDirectory.createFolder();
        }

        //
        // Copy template
        //

        // TODO make customizable

        ZipUtils.extractZip(new ClasspathResource(classResolver,
                "/com/erinors/tapestry/tapdoc/service/xdoc/resources.zip"), outputDirectory);

        //
        // Generate component docs
        //

        for (Library library : libraryModels)
        {
            String libraryName = library.getName();
            String libraryLocation = library.getLocation();

            generatedOutputDirectory.resolveFile(fileNameGenerator.getLibraryDirectory(libraryLocation)).createFolder();

            try
            {
                String result = XsltUtils.xsltTransform(tapdocXml, getClass().getResourceAsStream("Library.xsl"),
                        "libraryName", libraryName);
                FileObject index = generatedOutputDirectory.resolveFile(
                        fileNameGenerator.getLibraryDirectory(libraryLocation)).resolveFile("index.xml");
                Writer out = new OutputStreamWriter(index.getContent().getOutputStream(), "UTF-8");
                out.write(result);
                out.close();
            }
            catch (Exception e)
            {
                throw new RuntimeException(e);
            }

            for (Component component : library.getComponents())
            {
                String componentName = component.getName();

                System.out.println("Generating " + libraryName + ":" + componentName + "...");

                Map<String, String> parameters = new HashMap<String, String>();
                parameters.put("libraryName", libraryName);
                parameters.put("componentName", componentName);

                String result = XsltUtils.xsltTransform(tapdocXml, getClass().getResourceAsStream("Component.xsl"),
                        parameters);

                Writer out = null;
                try
                {
                    FileObject index = generatedOutputDirectory.resolveFile(fileNameGenerator.getComponentIndexFile(
                            libraryLocation, componentName, true));
                    out = new OutputStreamWriter(index.getContent().getOutputStream(), "UTF-8");
                    out.write(result);
                    out.close(); // FIXME finally

                    // copy of html resources directory

                    Resource specificationLocation = component.getSpecificationLocation();
                    if (specificationLocation.getRelativeResource(componentName + "_tapdoc/resource").getResourceURL() != null)
                    {
                        File srcResourcesDirectory = new File(specificationLocation.getRelativeResource(
                                componentName + "_tapdoc/resource").getResourceURL().toURI());
                        FileObject dstResourcesFileObject = outputDirectory.resolveFile(
                                fileNameGenerator.getComponentDirectory(libraryLocation, componentName)).resolveFile(
                                "resource");
                        if (srcResourcesDirectory.exists() && srcResourcesDirectory.isDirectory())
                        {
                            File[] files = srcResourcesDirectory.listFiles();

                            if (files != null)
                            {
                                for (File resource : files)
                                {
                                    if (resource.isFile() && !resource.isHidden()) // FIXME handle directories
                                    {
                                        FileObject resourceFileObject = dstResourcesFileObject.resolveFile(resource
                                                .getName());
                                        resourceFileObject.createFile();

                                        InputStream inResource = null;
                                        OutputStream outResource = null;

                                        try
                                        {
                                            inResource = new FileInputStream(resource);
                                            outResource = resourceFileObject.getContent().getOutputStream();
                                            IOUtils.copy(inResource, outResource);
                                        }
                                        finally
                                        {
                                            IOUtils.closeQuietly(inResource);
                                            IOUtils.closeQuietly(outResource);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    throw new RuntimeException(e); // TODO
                }
                finally
                {
                    IOUtils.closeQuietly(out);
                }
            }
        }

        //
        // Create index file
        //

        {
            Writer out = null;
            try
            {
                String result = XsltUtils.xsltTransform(tapdocXml, getClass().getResourceAsStream("Overview.xsl"));
                FileObject index = generatedOutputDirectory.resolveFile("index.xml");
                out = new OutputStreamWriter(index.getContent().getOutputStream(), "UTF-8");
                out.write(result);
                out.close();
            }
            catch (Exception e)
            {
                throw new RuntimeException(e); // TODO
            }
            finally
            {
                IOUtils.closeQuietly(out);
            }
        }
    }

    public String getGeneratedFileNameExtension()
    {
        return "xml";
    }
}
