//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc.service;

import com.erinors.tapestry.tapdoc.model.Component;
import com.erinors.tapestry.tapdoc.model.Library;
import com.erinors.tapestry.tapdoc.model.Parameter;
import com.erinors.tapestry.tapdoc.xml.XmlCdata;
import com.erinors.tapestry.tapdoc.xml.XmlElement;
import com.erinors.tapestry.tapdoc.xml.XmlPart;
import com.erinors.tapestry.tapdoc.xml.XmlText;

/**
 * @author Norbert Sándor
 */
public class XmlGeneratorImpl implements XmlGenerator
{
    public XmlGeneratorImpl(DocResolver docResolver)
    {
        this.docResolver = docResolver;
    }

    private final DocResolver docResolver;

    public XmlPart generateXml(Iterable<Library> libraries)
    {
        XmlElement doc = new XmlElement("tapdoc");

        for (Library library : libraries)
        {
            doc.add(provideXml(library));
        }

        return doc;
    }

    protected XmlPart provideXml(Library libraryModel)
    {
        String libraryName = libraryModel.getName();

        XmlElement library = new XmlElement("library");
        library.addAttribute("name", libraryName);
        library.addAttribute("link", docResolver.getLibraryUrl(libraryModel.getLocation()));

        if (libraryModel.getDescription() != null)
        {
            XmlElement description = new XmlElement("description");
            library.add(description);
            // TODO parse
            description.add(new XmlText(libraryModel.getDescription()));
        }

        for (Component componentModel : libraryModel.getComponents())
        {
            library.add(provideXml(componentModel, libraryModel));
        }

        // FIXME pages

        return library;
    }

    protected XmlPart provideXml(Component componentModel, Library libraryModel)
    {
        XmlElement component = new XmlElement("component");
        component.addAttribute("name", componentModel.getName());
        component.addAttribute("link", docResolver
                .getComponentUrl(libraryModel.getLocation(), componentModel.getName()));

        component.addAttribute("allow-body", Boolean.toString(componentModel.getAllowBody()));
        component.addAttribute("allow-informal-parameters", Boolean.toString(componentModel
                .getAllowInformalParameters()));
        component.addAttribute("deprecated", Boolean.toString(componentModel.isDeprecated()));
        component.addAttribute("global", Boolean.toString(componentModel.isGlobal()));

        XmlElement classNameTag = new XmlElement("class");
        component.add(classNameTag);
        classNameTag.addAttribute("name", componentModel.getComponentClassName());
        classNameTag.addAttribute("url", docResolver.getJavadocUrl(componentModel.getComponentClassName(), null));

        if (componentModel.getHasSeeAlsoReferences())
        {
            XmlElement seeAlso = new XmlElement("see-also");
            component.add(seeAlso);

            for (String referencedComponent : componentModel.getSeeAlsoComponents())
            {
                if (referencedComponent.contains(":"))
                {
                    // FIXME handle
                }
                else
                {
                    seeAlso.add(new XmlElement("component", "label", referencedComponent, "library", libraryModel
                            .getName(), "component", referencedComponent, "link", docResolver.getComponentUrl(
                            libraryModel.getLocation(), referencedComponent)));
                }
            }

            for (String javadocReference : componentModel.getSeeAlsoClasses())
            {
                String typeName = javadocReference.contains("#") ? javadocReference.substring(0, javadocReference
                        .indexOf('#')) : javadocReference;
                String memberName = javadocReference.contains("#") ? javadocReference.substring(javadocReference
                        .indexOf('#') + 1) : null;

                String javadocUrl = docResolver.getJavadocUrl(typeName, memberName);

                seeAlso.add(new XmlElement("java", "label", javadocReference, "url", javadocUrl));
            }
        }

        if (componentModel.getVisuality() != null)
        {
            switch (componentModel.getVisuality())
            {
                case Visual:
                {
                    XmlElement visual = new XmlElement("visual");
                    component.add(visual);

                    visual.add(new XmlCdata(componentModel.getVisualSample()));
                }
                    break;
                case VisualOrNonvisual:
                {
                    XmlElement visual = new XmlElement("visual-or-nonvisual");
                    component.add(visual);

                    visual.add(new XmlCdata(componentModel.getVisualSample()));
                }
                    break;
                case Nonvisual:
                {
                    XmlElement nonvisual = new XmlElement("nonvisual");
                    component.add(nonvisual);
                }
                    break;
            }
        }

        if (componentModel.getDescription() != null)
        {
            XmlElement description = new XmlElement("description");
            component.add(description);
            // TODO parse description
            description.add(new XmlText(componentModel.getDescription()));
        }

        for (String reservedParameter : componentModel.getReservedParameters())
        {
            component.add(new XmlElement("reserved-parameter", "name", reservedParameter));
        }

        for (Parameter parameterModel : componentModel.getParameters())
        {
            // TODO külön providexml()-be a paraméter előállítás
            XmlElement parameter = new XmlElement("parameter");
            component.add(parameter);

            parameter.addAttribute("name", parameterModel.getName());
            parameter.addAttribute("required", Boolean.toString(parameterModel.isRequired()));

            if (parameterModel.getDefaultValue() != null)
            {
                parameter.addAttribute("default-value", parameterModel.getDefaultValue());
            }

            String type = parameterModel.getType();

            XmlElement typeTag = new XmlElement("type");
            parameter.add(typeTag);
            typeTag.addAttribute("name", Object.class.getName().equals(type) ? "object" : type);
            typeTag.addAttribute("url", docResolver.getJavadocUrl(type, null));

            parameter.addAttribute("cache", Boolean.toString(parameterModel.isCached()));

            for (String alias : parameterModel.getAliases())
            {
                parameter.add(new XmlElement("alias", "name", alias));
            }

            if (parameterModel.getDescription() != null)
            {
                XmlElement paramDescription = new XmlElement("description");
                parameter.add(paramDescription);
                // TODO parse
                paramDescription.add(new XmlText(parameterModel.getDescription()));
            }
        }

        if (componentModel.getHtmlAddon() != null)
        {
            XmlElement htmlAddon = new XmlElement("html-addon");
            component.add(htmlAddon);

            htmlAddon.add(new XmlCdata(componentModel.getHtmlAddon()));
        }

        return component;
    }
}
