//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc.service;

import java.io.File;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.digester.Digester;
import org.apache.hivemind.ApplicationRuntimeException;
import org.apache.hivemind.ClassResolver;
import org.apache.hivemind.HiveMind;
import org.apache.hivemind.Resource;
import org.apache.hivemind.util.ClasspathResource;
import org.apache.tapestry.BaseComponent;
import org.apache.tapestry.INamespace;
import org.apache.tapestry.asset.AssetSource;
import org.apache.tapestry.engine.ISpecificationSource;
import org.apache.tapestry.engine.Namespace;
import org.apache.tapestry.pageload.ComponentClassProvider;
import org.apache.tapestry.pageload.ComponentClassProviderContext;
import org.apache.tapestry.resolver.ComponentSpecificationResolver;
import org.apache.tapestry.services.ComponentConstructorFactory;
import org.apache.tapestry.services.NamespaceResources;
import org.apache.tapestry.services.impl.NamespaceResourcesImpl;
import org.apache.tapestry.spec.IComponentSpecification;
import org.apache.tapestry.spec.ILibrarySpecification;
import org.apache.tapestry.spec.IParameterSpecification;

import com.erinors.tapestry.tapdoc.model.Component;
import com.erinors.tapestry.tapdoc.model.Library;
import com.erinors.tapestry.tapdoc.model.Parameter;

/**
 * @author Norbert Sándor
 */
public class ModelSourceImpl implements ModelSource
{
    public ModelSourceImpl(TapdocContext tapdocContext, ClassResolver classResolver,
            FileNameGenerator fileNameGenerator, AssetSource assetSource,
            ComponentSpecificationResolver componentSpecificationResolver,
            ComponentConstructorFactory componentConstructorFactory, JavadomParser javadocParser)
    {
        this.tapdocContext = tapdocContext;
        this.classResolver = classResolver;
        this.fileNameGenerator = fileNameGenerator;
        this.assetSource = assetSource;
        this.componentSpecificationResolver = componentSpecificationResolver;
        this.componentConstructorFactory = componentConstructorFactory;
        this.javadocParser = javadocParser;
    }

    private final TapdocContext tapdocContext;

    private final ClassResolver classResolver;

    private final FileNameGenerator fileNameGenerator;

    private final AssetSource assetSource;

    private final ComponentSpecificationResolver componentSpecificationResolver;

    private final ComponentConstructorFactory componentConstructorFactory;

    private final JavadomParser javadocParser;

    private ComponentClassProvider componentClassProvider;

    public void setComponentClassProvider(ComponentClassProvider componentClassProvider)
    {
        this.componentClassProvider = componentClassProvider;
    }

    private ISpecificationSource specificationSource;

    public void setSpecificationSource(ISpecificationSource specificationSource)
    {
        this.specificationSource = specificationSource;
    }

    private NamespaceResources namespaceResources;

    /**
     * Initializer method called by Hivemind.
     */
    public void initialize()
    {
        namespaceResources = new NamespaceResourcesImpl(specificationSource, assetSource);
    }

    public Library getLibraryModel(String libraryLocation, boolean ignoreAbstract)
    {
        Resource location = new ClasspathResource(classResolver, libraryLocation);

        ILibrarySpecification specification = libraryLocation.endsWith(".application") ? specificationSource
                .getApplicationNamespace().getSpecification() : specificationSource.getLibrarySpecification(location);
        INamespace namespace = new Namespace(null, null, specification, namespaceResources);

        Library library = new Library(specification.getComponentTypes());
        library.setName(fileNameGenerator.extractLibraryName(libraryLocation));
        library.setLocation(libraryLocation);

        library.setDescription(specification.getDescription());

        //
        // Read .tapdoc.xml
        //

        Resource extLibraryDocResource = location.getRelativeResource(location.getName().substring(0,
                location.getName().lastIndexOf('.'))
                + ".tapdoc.xml");

        if (extLibraryDocResource != null && extLibraryDocResource.getResourceURL() != null)
        {
            Digester digester = new Digester();
            digester.setValidating(false);

            digester.push(library);

            digester.addCallMethod("tapdoc/documented", "addDocumentedComponent", 1);
            digester.addCallParam("tapdoc/documented", 0);

            try
            {
                digester.parse(new InputStreamReader(extLibraryDocResource.getResourceURL().openStream(), "UTF-8"));
            }
            catch (Exception e)
            {
                throw new ApplicationRuntimeException("Cannot process " + extLibraryDocResource, e);
            }
        }

        List<String> components = library.getComponentTypes();
        components.addAll(javadocParser.getAnnotatedComponent(libraryLocation, ignoreAbstract));

        for (String componentName : components)
        {
            componentSpecificationResolver.resolve(null, namespace, componentName, null);
            // FIXME komponens nevét is a resolver-től szedje!
            IComponentSpecification componentSpecification = componentSpecificationResolver.getSpecification();

            // NOTE Copied from org.apache.tapestry.pageload.PageLoader(release 4.0) with some modifications
            // BEGIN
            ComponentClassProviderContext context = new ComponentClassProviderContext("PLACEHOLDER",
                    componentSpecification, namespace);
            String componentClassName = componentClassProvider.provideComponentClassName(context);

            Class<?> componentClass;
            if (HiveMind.isBlank(componentClassName))
            {
                componentClass = BaseComponent.class;
            }
            else
            {
                componentClass = classResolver.findClass(componentClassName);
            }

            try {
                componentConstructorFactory.getComponentConstructor(componentSpecification, componentClassName);
            } catch (Throwable ignore) { }

            // END

            Component component = new Component();
            library.getComponents().add(component);
            
            component.setName(componentName);
            component.setSpecificationLocation(componentSpecification.getLocation().getResource());

            List<String> globalComponents = Collections.emptyList();
            try
            {
                globalComponents = tapdocContext.getRegistry().getConfiguration(
                        "com.erinors.tapestry.nsutils.GlobalComponents");
            }
            catch (Exception e)
            {
            }

            // TODO support @Global
            // component.setGlobal(globalComponents.contains(componentSpecification.getLocation().getResource().getPath())
            //        || componentClass.isAnnotationPresent(GlobalComponent.class));

            component.setAllowBody(componentSpecification.getAllowBody());
            component.setAllowInformalParameters(componentSpecification.getAllowInformalParameters());
            component.setComponentClassName(componentClassName);
            component.setDeprecated(componentSpecification.isDeprecated());

            if (componentSpecification.getDescription() != null)
            {
                component.setDescription(componentSpecification.getDescription());
            }
            else
            {
                component.setDescription(javadocParser.getClassComment(componentClassName));
            }

            //
            // Read .tapdoc.xml
            //

            Resource extDocResource = componentSpecification.getLocation().getResource().getRelativeResource(
                    componentName + ".tapdoc.xml");
            if (extDocResource != null && extDocResource.getResourceURL() != null)
            {
                Digester digester = new Digester();
                digester.setValidating(false);

                digester.push(component);

                digester.addCallMethod("tapdoc/see-also/component", "addSeeAlsoComponent", 1);
                digester.addCallParam("tapdoc/see-also/component", 0);

                digester.addCallMethod("tapdoc/see-also/java", "addSeeAlsoClass", 1);
                digester.addCallParam("tapdoc/see-also/java", 0);

                digester.addCallMethod("tapdoc/visual", "setVisual", 1);
                digester.addCallParam("tapdoc/visual", 0);

                digester.addCallMethod("tapdoc/nonvisual", "setNonvisual");

                digester.addCallMethod("tapdoc/visual-or-nonvisual", "setVisualOrNonvisual", 1);
                digester.addCallParam("tapdoc/visual-or-nonvisual", 0);

                digester.addCallMethod("tapdoc/html-addon", "setHtmlAddon", 1);
                digester.addCallParam("tapdoc/html-addon", 0);

                // FIXME hivemind referencia, általános url

                try
                {
                    digester.parse(new InputStreamReader(extDocResource.getResourceURL().openStream(), "UTF-8"));
                }
                catch (Exception e)
                {
                    throw new ApplicationRuntimeException("Cannot process " + extDocResource, e);
                }
            }

            // search for html addon resources directory (<componentName>.tapdoc.resources)

            Resource extResourcesDir = componentSpecification.getLocation().getResource().getRelativeResource(
                    componentName + ".tapdoc.resources");
            if (extResourcesDir != null && extResourcesDir.getResourceURL() != null)
            {
                try
                {

                    File file = new File(extResourcesDir.getResourceURL().toURI());
                    if (file.exists() && file.isDirectory())
                    {
                        // FIXME component.setHtmlResourcesDir(extResourcesDir);
                    }

                }
                catch (URISyntaxException e)
                {
                    throw new ApplicationRuntimeException("Cannot process " + extResourcesDir + ". " + e.toString(), e);
                }
            }

            // FIXME
            /*
             * // read html addon file (<componentName>.tapdoc.html) Resource extDocHtml =
             * componentSpecification.getLocation().getResource().getRelativeResource( componentName + ".tapdoc.html");
             * if (extDocHtml != null && extDocHtml.getResourceURL() != null) { try { InputStream htmlInputStream =
             * extDocHtml.getResourceURL().openStream();
             * 
             * String html = IOUtils.toString(htmlInputStream); // readContent(inputStream); htmlInputStream.close();
             * 
             * if (component.getHtmlResourcesDir() != null) html = html.replaceAll(componentName + ".tapdoc.resources",
             * "resources");
             * 
             * component.setHtmlAddon(html); } catch (IOException e) { throw new ApplicationRuntimeException("Cannot
             * process " + extDocResource, e); } }
             */

            if (componentSpecification.getReservedParameterNames() != null)
            {
                for (String reservedParameter : (Set<String>) componentSpecification.getReservedParameterNames())
                {
                    boolean formal = false;
                    for (String formalParam : (List<String>) componentSpecification.getParameterNames())
                    {
                        if (reservedParameter.equals(formalParam.toLowerCase()))
                        {
                            formal = true;
                            break;
                        }
                    }

                    if (!formal)
                    {
                        component.getReservedParameters().add(reservedParameter);
                    }
                }
            }

            if (componentSpecification.getParameterNames() != null)
            {
                for (String parameterName : (List<String>) componentSpecification.getParameterNames())
                {
                    IParameterSpecification paramSpec = componentSpecification.getParameter(parameterName);
                    String type = paramSpec.getType();

                    Parameter parameter = new Parameter();
                    parameter.setName(parameterName);
                    component.getParameters().add(parameter);

                    parameter.setRequired(paramSpec.isRequired());
                    parameter.setDeprecated(paramSpec.isDeprecated());
                    parameter.setAliases(new HashSet<String>(paramSpec.getAliasNames()));

                    parameter.setDescription(paramSpec.getDescription());

                    try
                    {
                        Method getter = null;
                        Method setter = null;

                        String tail = Character.toUpperCase(paramSpec.getPropertyName().charAt(0))
                                + paramSpec.getPropertyName().substring(1);

                        try
                        {
                            String getterName = "get" + tail;
                            getter = componentClass.getMethod(getterName, (Class[]) null);
                        }
                        catch (NoSuchMethodException e)
                        {
                            try
                            {
                                String boolGetterName = "is" + tail;
                                getter = componentClass.getMethod(boolGetterName, (Class[]) null);
                            }
                            catch (NoSuchMethodException e1)
                            {
                            }
                        }

                        String setterName = "set" + tail;
                        for (Method method : componentClass.getMethods())
                        {
                            if (method.getName().equals(setterName) && method.getReturnType() == void.class
                                    && method.getParameterTypes().length == 1)
                            {
                                setter = method;
                                break;
                            }
                        }

                        if (getter != null)
                        {
                            type = getter.getReturnType().getName();
                        }
                        else if (setter != null)
                        {
                            type = setter.getParameterTypes()[0].getName();
                        }

                        if (parameter.getDescription() == null && (getter != null || setter != null))
                        {
                            parameter.setDescription(javadocParser.getAccessorMethodComment(getter, setter));
                        }
                    }
                    catch (Exception e)
                    {
                        throw new RuntimeException(e);
                    }

                    if (type == null)
                    {
                        // NOTE if the parameter is defined in the specification using <parameter> tag and it doesn't
                        // have an accessor method then type is null in Tapestry4.
                        type = Object.class.getName();
                    }

                    parameter.setType(type);

                    if (paramSpec.getDefaultValue() != null)
                    {
                        parameter.setDefaultValue(paramSpec.getDefaultValue());
                    }
                    else if (!parameter.isRequired())
                    {
                        String implicitDefaultValue = null;

                        if (type.equals("boolean"))
                        {
                            implicitDefaultValue = "false";
                        }
                        else if (type.equals("char"))
                        {
                            implicitDefaultValue = "'\\u0000'";
                        }
                        else if (type.equals("byte") || type.equals("short") || type.equals("int"))
                        {
                            implicitDefaultValue = "0";
                        }
                        else if (type.equals("long"))
                        {
                            implicitDefaultValue = "0l";
                        }
                        else if (type.equals("float"))
                        {
                            implicitDefaultValue = "0.0f";
                        }
                        else if (type.equals("double"))
                        {
                            implicitDefaultValue = "0.0d";
                        }

                        parameter.setDefaultValue(implicitDefaultValue);
                    }
                }
            }
        }

        return library;
    }
}
