//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc.service;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * @author Norbert Sándor
 */
public class JavadomParserImpl implements JavadomParser
{
    public JavadomParserImpl(TapdocContext context, DocResolver docResolver)
    {
        this.context = context;
        this.docResolver = docResolver;
    }

    private final TapdocContext context;

    private final DocResolver docResolver;

    private XPath xpath;

    private Document javadomDocument;

    private boolean initialized;

    private void initialize()
    {
        if (!initialized)
        {
            XPathFactory xpathFactory = XPathFactory.newInstance();
            xpath = xpathFactory.newXPath();

            try
            {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setValidating(false);

                javadomDocument = factory.newDocumentBuilder().parse(
                        new InputSource(context.getJavaDom().getContent().getInputStream()));
            }
            catch (Exception e)
            {
                throw new RuntimeException(e); // FIXME
            }

            initialized = true;
        }
    }

    public String getClassComment(String className)
    {
        initialize();

        String classComment = null;
        try
        {
            XPathExpression expression = xpath.compile("/tapdoc-javadom/java-class[@fulltype='" + className
                    + "']/comment/inlineTags/*");
            classComment = processComment(expression);
        }
        catch (Exception e)
        {
            throw new RuntimeException(e); // FIXME
        }

        return classComment;
    }

    public String getAccessorMethodComment(Method getter, Method setter)
    {
        initialize();

        Method accessor = getter != null ? getter : setter;

        String accessorComment = null;

        if (accessor != null)
        {
            try
            {
                XPathExpression expression = xpath.compile("/tapdoc-javadom/java-class[@fulltype='"
                        + accessor.getDeclaringClass().getName() + "']/methods/method[@name='" + accessor.getName()
                        + "']/comment/inlineTags/*");
                accessorComment = processComment(expression);
            }
            catch (Exception e)
            {
                throw new RuntimeException(e); // FIXME
            }
        }

        return accessorComment;
    }

    private String processComment(XPathExpression expression) throws XPathExpressionException
    {
        String comment = null;

        NodeList inlineNodes = (NodeList) expression.evaluate(javadomDocument, XPathConstants.NODESET);

        if (inlineNodes.getLength() > 0)
        {
            comment = "";
        }

        for (int i = 0; i < inlineNodes.getLength(); i++)
        {
            Node inlineTag = inlineNodes.item(i);

            if (inlineTag.getChildNodes().getLength() > 0)
            {
                boolean link = inlineTag.getAttributes().getNamedItem("name") != null
                        && "@link".equals(inlineTag.getAttributes().getNamedItem("name").getNodeValue());

                Node text = inlineTag.getChildNodes().item(0); // FIXME

                boolean disabled = false;
                if (link)
                {
                    Node classNode = inlineTag.getAttributes().getNamedItem("class");
                    Node memberNode = inlineTag.getAttributes().getNamedItem("member");

                    String url = docResolver.getJavadocUrl(classNode != null ? classNode.getNodeValue() : null,
                            memberNode != null ? memberNode.getNodeValue() : null);
                    disabled = url == null;

                    if (disabled)
                    {
                        comment += "<i>";
                    }
                    else
                    {
                        comment += "<a href=\"";
                        comment += url;
                        comment += "\">";
                    }
                }

                comment += text.getTextContent();

                if (link)
                {
                    comment += disabled ? "</i>" : "</a>";
                }
            }
        }

        if (comment != null && comment.length() == 0)
        {
            comment = null;
        }

        return comment;
    }

    public List<String> getAnnotatedComponent(String library, boolean ignoreAbstract) {
        initialize();
        
        int pos = library.lastIndexOf("/");
        library = library.substring(0, pos).replace('/', '.');
        if (library.startsWith("."))
            library = library.substring(1);

        List<String> components = new ArrayList<String>();
        NodeList inlineNodes;
        
        try {
            XPathExpression expression = xpath.compile("/tapdoc-javadom/java-class");
            inlineNodes = (NodeList) expression.evaluate(javadomDocument, XPathConstants.NODESET);
        } catch (XPathExpressionException e) {
            return components;
        }

        for (int i = 0; i < inlineNodes.getLength(); i++) {
            NamedNodeMap attrs = inlineNodes.item(i).getAttributes();
            String fulltype = attrs.getNamedItem("fulltype").getNodeValue();
            boolean isAbstract = "true".equals(attrs.getNamedItem("abstract").getNodeValue());
            boolean isComponent = (attrs.getNamedItem("tapestry-component") != null);
            if (!fulltype.startsWith(library)) {
                continue;
            }

            String type = fulltype.substring(library.length() + 1);
            type = type.replace('.', '/');

            if (isComponent || (isAbstract && !ignoreAbstract)) {
                components.add(type);
            }
        }
        return components;
    }
}