//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc.doclet;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;

import org.apache.tapestry.IMarkupWriter;
import org.apache.tapestry.markup.MarkupWriterImpl;
import org.apache.tapestry.markup.UTFMarkupFilter;

import com.erinors.tapestry.tapdoc.xml.XmlElement;
import com.erinors.tapestry.tapdoc.xml.XmlText;
import com.sun.javadoc.AnnotationDesc;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.Doc;
import com.sun.javadoc.DocErrorReporter;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.Parameter;
import com.sun.javadoc.RootDoc;
import com.sun.javadoc.SeeTag;
import com.sun.javadoc.Tag;

/**
 * Custom Javadoc doclet for Tapdoc annotation support. Used to extract comments from Java source code.
 * 
 * @author Norbert Sándor
 */
public class TapdocDoclet
{
    private static String outputDirectory = "./";

    private static final String GlobalComponentAnnotation = "com.erinors.tapestry.nsutils.GlobalComponent";

    public static boolean start(RootDoc root)
    {
        //
        // Get options
        //

        String[][] options = root.options();
        for (int i = 0; i < options.length; i++)
        {
            if (options[i][0].compareToIgnoreCase("-d") == 0)
            {
                outputDirectory = options[i][1];

                String separator = System.getProperty("file.separator");
                if (!outputDirectory.endsWith(separator))
                {
                    outputDirectory += separator;
                }
            }
        }

        //
        // Generate doc
        //

        XmlElement rootNode = new XmlElement("tapdoc-javadom");

        for (ClassDoc classDoc : root.classes())
        {
            rootNode.add(processClass(classDoc));
        }

        try
        {
            File outputFile = new File(outputDirectory, "tapdoc-javadom.xml");
            IMarkupWriter out = new MarkupWriterImpl("text/xml", new PrintWriter(outputFile, "utf-8"),
                    new UTFMarkupFilter());

            rootNode.toXml(out);
            out.close();
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot write XML output.", e);
        }

        return true;
    }

    public static int optionLength(String option)
    {
        int length;

        if (option.compareToIgnoreCase("-d") == 0)
        {
            return 2;
        }
        else
        {
            length = 0;
        }

        return length;
    }

    public static boolean validOptions(String options[][], DocErrorReporter reporter)
    {
        return true;
    }

    private static XmlElement processClass(ClassDoc classDoc)
    {
        XmlElement classNode = new XmlElement("java-class");

        classNode.addAttribute("type", classDoc.name());
        classNode.addAttribute("fulltype", classDoc.qualifiedName());
        classNode.addAttribute("package", classDoc.containingPackage().name());
        classNode.addAttribute("abstract", String.valueOf(classDoc.isAbstract()));

        if (classDoc.superclass() != null)
        {
            classNode.addAttribute("superclass", classDoc.superclass().name());
            classNode.addAttribute("superclass-fulltype", classDoc.superclass().qualifiedName());
        }

        for (AnnotationDesc annotationDesc : classDoc.annotations())
        {
            if ("ComponentClass".equals(annotationDesc.annotationType().name()))
            {
                classNode.addAttribute("tapestry-component", "true");
                break;
            }
        }

        // FIXME this should be uncommented but causes an error
        /*
         * for (AnnotationDesc annotationDesc : classDoc.annotations()) { if
         * (GlobalComponentAnnotation.equals(annotationDesc.annotationType().qualifiedName())) {
         * classNode.addAttribute("global", "true"); break; } }
         */

        classNode.add(processComment(classDoc));

        XmlElement methodsNode = new XmlElement("methods");
        classNode.add(methodsNode);
        for (MethodDoc methodDoc : classDoc.methods())
        {
            methodsNode.add(processMethod(methodDoc));
        }

        for (ClassDoc innerClassDoc : classDoc.innerClasses())
        {
            classNode.add(processClass(innerClassDoc));
        }

        return classNode;
    }

    private static XmlElement processMethod(MethodDoc methodDoc)
    {
        XmlElement methodNode = new XmlElement("method");
        methodNode.addAttribute("name", methodDoc.name());

        methodNode.add(processComment(methodDoc));

        XmlElement parameters = new XmlElement("parameters");
        methodNode.add(parameters);
        for (Parameter parameter : methodDoc.parameters())
        {
            XmlElement parameterNode = new XmlElement("parameter");
            parameterNode.add(parameterNode);

            parameterNode.addAttribute("name", parameter.name());
            parameterNode.addAttribute("type", parameter.type().typeName());
            parameterNode.addAttribute("fulltype", parameter.type().qualifiedTypeName());
        }

        methodNode.addAttribute("type", methodDoc.returnType().typeName());
        methodNode.addAttribute("fulltype", methodDoc.returnType().qualifiedTypeName());

        return methodNode;
    }

    private static XmlElement processComment(Doc doc)
    {
        XmlElement commentNode = new XmlElement("comment");

        if (doc.inlineTags() != null && doc.inlineTags().length > 0)
        {
            XmlElement inlineTags = new XmlElement("inlineTags");
            commentNode.add(inlineTags);

            for (Tag inlineTag : doc.inlineTags())
            {
                XmlElement inlineNode;

                if (inlineTag instanceof SeeTag)
                {
                    SeeTag seeTag = (SeeTag) inlineTag;

                    inlineNode = new XmlElement("see");

                    if (seeTag.referencedClassName() != null)
                    {
                        inlineNode.addAttribute("class", seeTag.referencedClassName());
                    }

                    if (seeTag.referencedMemberName() != null)
                    {
                        inlineNode.addAttribute("member", seeTag.referencedMemberName());
                    }

                    if (seeTag.label() != null && seeTag.label().length() > 0)
                    {
                        inlineNode.add(new XmlText(seeTag.label()));
                    }
                    else
                    {
                        if (seeTag.referencedClassName() != null)
                        {
                            inlineNode.add(new XmlText(seeTag.referencedClassName()));
                            // TODO use unqualified name if in the same package
                        }

                        if (seeTag.referencedMemberName() != null)
                        {
                            if (seeTag.referencedClassName() != null)
                            {
                                inlineNode.add(new XmlText("."));
                            }

                            inlineNode.add(new XmlText(seeTag.referencedMemberName()));
                        }
                    }
                }
                else
                {
                    inlineNode = new XmlElement("text");
                    inlineNode.add(new XmlText(inlineTag.text()));
                }

                if (inlineNode != null)
                {
                    inlineNode.addAttribute("name", inlineTag.name());
                    inlineTags.add(inlineNode);
                }
            }
        }

        return commentNode.getBody().size() > 0 ? commentNode : null;
    }
}
