//
// Copyright (c) Erinors 2006-2007
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.erinors.tapestry.tapdoc;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.commons.vfs.FileSystemManager;
import org.apache.commons.vfs.VFS;
import org.apache.hivemind.ApplicationRuntimeException;
import org.apache.hivemind.ClassResolver;
import org.apache.hivemind.impl.DefaultClassResolver;
import org.apache.hivemind.impl.RegistryBuilder;
import org.apache.tools.ant.Task;

import com.erinors.tapestry.tapdoc.util.TapdocPrefs;
import com.erinors.tapestry.tapdoc.util.TapdocUtils;

/**
 * The main class for executing Tapdoc, either as an Ant task or as a Java application.
 * 
 * @author Norbert Sándor
 */
public class TapDocTask extends Task
{
    /**
     * Main method for executing Tapdoc. Arguments:
     * <ol>
     * <li>output directory</li>
     * <li>comma separated list of library locations, eg.
     * &quot;/org/apache/tapestry/Framework.library,/org/apache/tapestry/contrib/Contrib.library&quot;</li>
     * <li>optional comma separated Javadoc root links, eg.
     * &quot;http://java.sun.com/javase/6/docs/api/,http://tapestry.apache.org/tapestry4.1/tapestry-framework/apidocs/&quot;.
     * Javadoc compatible format is supported as well (&quot;-link url1 -link url2 ...&quot;).</li>
     * </ol>
     * 
     * @param args
     *            arguments
     */
    public static void main(String[] args)
    {
        if (args.length < 2)
        {
            printUsage();
        }
        else
        {
            TapDocTask tapDoc = new TapDocTask();

            tapDoc.setOutputDirectory(new File(args[0]));
            tapDoc.setGeneratedOutputDirectory(new File(args[0]));

            tapDoc.setLibraries(args[1]);

            if (args.length > 2)
            {
                tapDoc.setJavadocLinks(args[2]);
            }

            tapDoc.execute();
        }
    }

    private static void printUsage()
    {
        System.out
                .println("arguments: (output directory) (library locations separated by comma) [javadoc root links separated by comma]");
    }

    private ClassResolver classResolver = new DefaultClassResolver(RegistryBuilder.class.getClassLoader());

    public void setClassResolver(ClassResolver classResolver)
    {
        this.classResolver = classResolver;
    }

    private File outputDirectory;

    /**
     * Sets the directory where the documentation is generated to.
     * 
     * @param outputDirectory
     *            output directory
     */
    public void setOutputDirectory(File outputDirectory)
    {
        if (!outputDirectory.exists())
        {
            outputDirectory.mkdir();
        }
        else if (!outputDirectory.isDirectory())
        {
            throw new RuntimeException("Invalid output directory: must be a directory not a normal file");
        }

        this.outputDirectory = outputDirectory;
    }

    private File generatedOutputDirectory;

    public void setGeneratedOutputDirectory(File generatedOutputDirectory)
    {
        if (!generatedOutputDirectory.exists())
        {
            generatedOutputDirectory.mkdir();
        }
        else if (!generatedOutputDirectory.isDirectory())
        {
            throw new RuntimeException("Invalid output directory: must be a directory not a normal file");
        }

        this.generatedOutputDirectory = generatedOutputDirectory;
    }

    private List<String> libraryLocations;

    /**
     * Sets the list of Tapestry libraries to generate documentation for.
     * 
     * @param libraries
     *            comma separated list of library locations
     */
    public void setLibraries(String libraries)
    {
        if (libraries == null || libraries.length() == 0)
        {
            throw new ApplicationRuntimeException("Invalid library locations: " + libraries);
        }

        libraryLocations = Arrays.asList(libraries.split(","));
    }

    private List<String> javadocLinks;

    /**
     * Sets the external Javadoc references.
     * 
     * @param javadocLinks
     *            Javadoc API locations
     */
    public void setJavadocLinks(String javadocLinks)
    {
        if (javadocLinks.trim().startsWith("-link"))
        {
            StringBuilder buffer = new StringBuilder();
            StringTokenizer tokenizer = new StringTokenizer(javadocLinks, "-link");
            while (tokenizer.hasMoreTokens())
            {
                if (buffer.length() > 0)
                {
                    buffer.append(',');
                }

                buffer.append(tokenizer.nextToken().trim());
            }

            javadocLinks = buffer.toString();
        }

        this.javadocLinks = Arrays.asList(javadocLinks.split(","));
    }

    private String fileSystemId = "file";

    public void setFileSystemId(String fileSystemId)
    {
        this.fileSystemId = fileSystemId;
    }

    private File javaDomFile;

    public void setJavaDomFile(File javaDomFile)
    {
        this.javaDomFile = javaDomFile;
    }

    private List<String> tapdocLinks;

    public void setTapdocLinks(String tapdocLinks)
    {
        this.tapdocLinks = null; // FIXME
    }

    /**
     * Executes Tapdoc. The <code>outputDirectory</code> and the <code>libraries</code> must be set before calling
     * this method.
     */
    public void execute()
    {
        if (outputDirectory == null)
        {
            throw new ApplicationRuntimeException(
                    "The output directory should be defined in parameter 'outputDirectory'.");
        }

        if (libraryLocations == null)
        {
            throw new ApplicationRuntimeException("Library locations should be defined in parameter 'libraries'.");
        }

        try
        {
            FileSystemManager fsManager = VFS.getManager();
            FileObject outputDirectoryFileObject = fsManager.resolveFile(fileSystemId + "://"
                    + outputDirectory.getAbsolutePath());
            FileObject generatedOutputDirectoryFileObject = fsManager.resolveFile(fileSystemId + "://"
                    + generatedOutputDirectory.getAbsolutePath());
            FileObject javaDomFileObject = javaDomFile != null ? fsManager.resolveFile(fileSystemId + "://"
                    + javaDomFile.getAbsolutePath()) : null;

            TapdocPrefs prefs = new TapdocPrefs(true, true);
            prefs.setLibraryLocations(libraryLocations);
            prefs.setJavadocLinks(javadocLinks);
            prefs.setTapdocLinks(tapdocLinks);
            TapdocUtils.generate(classResolver, outputDirectoryFileObject, generatedOutputDirectoryFileObject,
                    javaDomFileObject, prefs);
        }
        catch (FileSystemException e)
        {
            throw new RuntimeException(e);
        }
    }
}
