//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package net.sf.tacos.annotations;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import org.apache.hivemind.ApplicationRuntimeException;
import org.apache.hivemind.Location;
import org.apache.hivemind.service.BodyBuilder;
import org.apache.hivemind.service.MethodSignature;
import org.apache.tapestry.annotations.MethodAnnotationEnhancementWorker;
import org.apache.tapestry.enhance.EnhancementOperation;
import org.apache.tapestry.enhance.EnhanceUtils;
import org.apache.tapestry.spec.IComponentSpecification;
import org.apache.tapestry.IComponent;

/**
 * @author Andreas Andreou
 */

public class CachedAnnotationWorker implements MethodAnnotationEnhancementWorker {

	public void performEnhancement(EnhancementOperation op, IComponentSpecification spec,
                Method method, Location location) {
        Class<?> type = method.getReturnType();
        if (type.equals(void.class))
                    throw new ApplicationRuntimeException(
                                    "Cached annotation cannot be used with a method that returns void");

        String fieldName = "_$cached$" + method.getName();
        MethodSignature signature = new MethodSignature(method);

        op.addField(fieldName, type);

        BodyBuilder builder = createCacheMethod(signature, fieldName);
        op.addMethod(Modifier.PUBLIC, signature, builder.toString(), location);

        extendCleanupAfterRender(op, fieldName);
    }

    BodyBuilder createCacheMethod(MethodSignature sig, String cacheField)
    {
        BodyBuilder result = new BodyBuilder();

        result.begin();

        result.addln("if (this.{0}==null)", cacheField);
        result.addln("this.{0} = super.{1}($$);", cacheField, sig.getName());

        result.addln("return this.{0};", cacheField);

        result.end();

        return result;
    }

    void extendCleanupAfterRender(EnhancementOperation op, String fieldName)
    {
        BodyBuilder cleanupBody = new BodyBuilder();
        cleanupBody.addln("{0} = null;", fieldName);
        op.extendMethodImplementation(IComponent.class,
                EnhanceUtils.CLEANUP_AFTER_RENDER_SIGNATURE, cleanupBody.toString());
    }
}