/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.lang;


import java.util.Arrays;
import java.util.Locale;


/**
 * Implementation of the MultiLingualText interface based on the use of the LanguageSupport.
 * 
 * @author Olivier Berlanger
 */
public class MultiLingualTextImpl implements MultiLingualText {

    private String key;
    private Object[] params;


    public MultiLingualTextImpl(String messageKey, Object... messageParams) {
        key = messageKey;
        params = messageParams;
        if ((params != null) && (params.length == 0)) params = null;
    }


    public String getRawText() {
        return (params == null) ? key : key + Arrays.toString(params);
    }


    public String getText() {
        String text;
        if (params == null) {
            text = LanguageSupport.getLocalizedString(key);
        } else {
            text = LanguageSupport.getLocalizedString(key, getTranslatedParams());
        }
        return text;
    }


    protected Object[] getTranslatedParams() {
        Object[] translatedParams = null;
        if (params != null) {
            int len = params.length;
            translatedParams = new Object[len];
            for (int i = 0; i < len; i++) {
                if (params[i] == null) {
                    translatedParams[i] = null;
                } else if (params[i] instanceof MultiLingualText) {
                    MultiLingualText multilingualParam = (MultiLingualText) params[i];
                    translatedParams[i] = multilingualParam.getText(LanguageSupport.getCurrentLocale());
                } else {
                    translatedParams[i] = params[i];
                }
            }
        }
        return translatedParams;
    }


    public String getText(Locale loc) {
        if ((loc == null) || loc.equals(LanguageSupport.getCurrentLocale())) {
            return getText();
        }
        throw new UnsupportedOperationException("Cannot get text for a locale (" + loc + ") that is not the current locale ("
                + LanguageSupport.getCurrentLocale() + ")");
    }


    public String getText(String localeId) {
        if ((localeId == null) || localeId.equals(LanguageSupport.getCurrentLocale().toString())) {
            return getText();
        }
        throw new UnsupportedOperationException("Cannot get text for a locale (" + localeId + ") that is not the current locale ("
                + LanguageSupport.getCurrentLocale() + ")");
    }

}
