/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.lang;


import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;


/**
 * Implementation of the MultiLingualText interface holding all possible translations in a Map.
 * 
 * @author Olivier Berlanger
 */
public class MultiLingualDirectText implements MultiLingualText {

    private static final String DEFAULT_TEXT_KEY = "default";

    private Map<String, String> translations;


    public MultiLingualDirectText(String defaultText) {
        translations = new HashMap<String, String>();
        translations.put(DEFAULT_TEXT_KEY, defaultText);
    }


    public void add(String localeId, String text) {
        if (localeId == null) localeId = DEFAULT_TEXT_KEY;
        translations.put(localeId, text);
    }


    public void remove(String localeId) {
        if (localeId == null) throw new IllegalArgumentException("Locale id cannot be null");
        translations.remove(localeId);
    }


    public String getText() {
        return getText(Locale.getDefault());
    }


    public String getText(Locale loc) {
        String text = null;
        if (loc != null) {
            text = translations.get(loc.toString());
            if (text == null) text = translations.get("" + loc.getLanguage() + "_" + loc.getCountry());
            if ((text == null) && (loc.getLanguage() != null)) text = translations.get(loc.getLanguage());
        }
        if (text == null) text = translations.get(DEFAULT_TEXT_KEY);
        return text;
    }


    public String getText(String localeId) {
        String text = null;
        if (localeId != null) text = translations.get(localeId);
        if (text == null) text = translations.get(DEFAULT_TEXT_KEY);
        return text;
    }


    public Iterator<String> getAvailableLocaleIds() {
        return translations.keySet().iterator();
    }

}
