/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.file;


import java.awt.Component;
import java.io.File;

import javax.swing.JFileChooser;

import net.sf.sfac.lang.LanguageSupport;
import net.sf.sfac.setting.Settings;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * Helper for file chooser usage. <br>
 * This helper will allow to:
 * <ul>
 * <li>Use settings to recall last used directory.
 * <li>Use the FileType(s) constants to specify chooser file types.
 * <li>Use LanguageSupport to translate FileChooser title & buttons.
 * </ul>
 * 
 * @author Olivier Berlanger
 */
public class FileChooserHelper {

    private static Log log = LogFactory.getLog(FileChooserHelper.class);
    private static final String DEFAULT_CHOOSER_DIRECTORY = "chooser.default.directory";

    private String id;
    private Settings setts;
    private JFileChooser chooser;
    private File preSelectedFile;
    private boolean useAllFilter;
    private FileType[] acceptedTypes;

    private String dialogTitleKey;
    private String dialogChooseKey;
    private String chooserDirectoryKey;


    public FileChooserHelper(String helperId, Settings settngs) {
        id = helperId;
        setts = settngs;
        chooser = new JFileChooser();
        dialogTitleKey = id + "_TITLE";
        dialogChooseKey = id + "_CHOOSE";
        chooserDirectoryKey = id + ".default.directory";
    }


    public int getFileSelectionMode() {
        return chooser.getFileSelectionMode();
    }


    public void setFileSelectionMode(int enabled) {
        chooser.setFileSelectionMode(enabled);
    }


    public void setDialogTitle(String title) {
        dialogTitleKey = title;
    }


    public String getDialogTitle() {
        return dialogTitleKey;
    }


    public void setPreSelectedFile(File f) {
        preSelectedFile = f;
    }


    public File getPreSelectedFile() {
        return preSelectedFile;
    }


    public void setAcceptedFileTypes(boolean useAll, FileType... types) {
        useAllFilter = useAll;
        acceptedTypes = types;
    }


    public File selectSingleFile(Component parent) {
        File selected = null;
        chooser.setMultiSelectionEnabled(false);
        setupChooserSelection();
        if (chooser.showOpenDialog(parent) == JFileChooser.APPROVE_OPTION) {
            selected = chooser.getSelectedFile();
            storeDefaultDirectorySetting();
        }
        return selected;
    }


    public File[] selectMultipleFiles(Component parent) {
        File[] selected = null;
        chooser.setMultiSelectionEnabled(true);
        setupChooserSelection();
        if (chooser.showOpenDialog(parent) == JFileChooser.APPROVE_OPTION) {
            selected = chooser.getSelectedFiles();
            storeDefaultDirectorySetting();
        }
        return selected;
    }


    private void setupChooserSelection() {
        // set customizable texts
        chooser.setDialogTitle(LanguageSupport.getOptionalLocalizedString(dialogTitleKey, "File selection"));
        chooser.setApproveButtonText(LanguageSupport.getOptionalLocalizedString(dialogChooseKey, "Select"));
        // set selection
        if (preSelectedFile == null) {
            String initialDirectory = setts.getFileProperty(chooserDirectoryKey, null);
            if (initialDirectory == null) initialDirectory = setts.getFileProperty(DEFAULT_CHOOSER_DIRECTORY, null);
            if (initialDirectory != null) {
                chooser.setCurrentDirectory(new File(initialDirectory));
            }
        } else {
            chooser.setSelectedFile(preSelectedFile);
        }
        // set filters
        if ((acceptedTypes != null) && (acceptedTypes.length > 0)) {
            addChoosableFileFilters();
        }
    }


    private void storeDefaultDirectorySetting() {
        String selectedPath = null;
        try {
            selectedPath = chooser.getCurrentDirectory().getAbsolutePath();
            setts.setFileProperty(chooserDirectoryKey, selectedPath);
            setts.setFileProperty(DEFAULT_CHOOSER_DIRECTORY, selectedPath);
        } catch (Exception e) {
            log.error("Cannot persist chooser current directory: " + selectedPath, e);
        }
    }


    /**
     * Add choosable file filter(s) to the given file chooser.
     * 
     * @param chooser
     *            the file chooser.
     * @param types
     *            the file types for which we want choosable file filter(s). (you can pass null if you want only set the 'select all
     *            (*.*)' filter)
     * @param addAllFilter
     *            true if the custom (= language aware) 'select all (*.*)' filter should be added.
     */
    private void addChoosableFileFilters() {
        if (useAllFilter) {
            chooser.setAcceptAllFileFilterUsed(false);
            chooser.addChoosableFileFilter(FileType.FILE_TYPE_ALL.getFileChooserFilter());
        }
        int len = (acceptedTypes == null) ? 0 : acceptedTypes.length;
        // reverse iterate so the first type (which is usually the default one) will be selected.
        for (int i = len - 1; i >= 0; i--) {
            chooser.addChoosableFileFilter(acceptedTypes[i].getFileChooserFilter());
        }
    }

}
