/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.editor;


import java.util.List;
import java.util.Locale;

import net.sf.sfac.editor.ValidationMessage.Level;
import net.sf.sfac.lang.LanguageSupport;
import net.sf.sfac.lang.MultiLingualText;
import net.sf.sfac.lang.MultiLingualTextImpl;


/**
 * Implementation of the ValidationReport interface throwing an exception at the first error. This report simply ignore all
 * validation message with level different from Level.ERROR.
 * 
 * @author Olivier Berlanger
 */
public class ExceptionValidationReport implements ValidationReport {

    public Level getGlobalValidationLevel() {
        return Level.OK;
    }


    public void addValidationMessage(ValidationMessage validationMsg) {
        if (validationMsg.getLevel() == Level.ERROR) {
            throw new ValidationException(validationMsg);
        }
    }


    public void addMandatoryError(String fieldPath, String fieldName) {
        MultiLingualTextImpl translatedName = new MultiLingualTextImpl(fieldName);
        addError(fieldPath, "MANDATORY_FIELD", translatedName);
    }


    public void addError(String fieldPath, String key, Object... params) {
        String path = fieldPath;
        addValidationMessage(new ValidationMessageImpl(Level.ERROR, path, key, params));
    }


    public void addWarning(String fieldPath, String key, Object... params) {
    }


    public void addInfo(String fieldPath, String key, Object... params) {
    }


    public List<ValidationMessage> getValidationMessages() {
        return null;
    }


    public ValidationReport getContextReport(String fieldPath) {
        return this;
    }

    static class ValidationException extends RuntimeException implements MultiLingualText {

        private static final long serialVersionUID = 1171045898175437661L;

        ValidationMessage message;


        public ValidationException(ValidationMessage msg) {
            message = msg;
        }


        @Override
        public String getMessage() {
            return message.getText(LanguageSupport.getCurrentLocale());
        }


        public ValidationMessage getValidationMessage() {
            return message;
        }


        public String getText() {
            return message.getText();
        }


        public String getText(Locale loc) {
            return message.getText(loc);
        }


        public String getText(String localeId) {
            return message.getText(localeId);
        }

    }


    public void removeValidationMessage(ValidationMessage validationMsg) {
        throw new UnsupportedOperationException("Not supported in " + getClass().getName());
    }


    public void removeAllValidationMessages() {
        throw new UnsupportedOperationException("Not supported in " + getClass().getName());
    }

}
