/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.utils;


import java.net.URL;

import javax.swing.ImageIcon;


public abstract class ResourceUtils {


    /**
     * Get the full name (including path) of a resource (absolute or relative to the given class). (Used mostly to display
     * correct error message because the classLoader can resolve this path itself).
     * 
     * @param baseClass
     *            used as base to resolve relative paths.
     * @param resourceName
     *            Path to the resource (absolute or relative to the given class).
     * @return the absolute resource path.
     */
    public static String getFullResourceName(Class<?> baseClass, String resourceName) {
        String packageName = baseClass.getPackage().getName();
        String basePath = packageName.replace('.', '/');
        return "/" + basePath + "/" + resourceName;
    }


    /**
     * Get an image icon resource with given path (absolute or relative to the given class).
     * 
     * @param baseClass
     *            used as base to resolve relative paths.
     * @param resourceName
     *            Path to the image resource (absolute or relative to the given class)
     * @return the requested image icon.
     * @exception IllegalArgumentException
     *                if there is no resource with the given name or if the resource is not an image.
     */
    public static ImageIcon getImageIcon(Class<?> baseClass, String resourceName) {
        ImageIcon icon;
        URL imageURL = baseClass.getResource(resourceName);
        if (imageURL == null)
            throw new IllegalArgumentException("Image not found in classpath: " + getFullResourceName(baseClass, resourceName));
        try {
            icon = new ImageIcon(imageURL);
        } catch (Exception e) {
            throw new IllegalArgumentException("Invalid image in URL: " + getFullResourceName(baseClass, resourceName)
                    + " -> exception: " + e);
        }
        return icon;
    }


}
