/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.utils;


import java.text.Collator;
import java.util.List;
import java.util.Locale;


public class Comparison {


    /**
     * Comparison of strings where a null string is considered as equivalent to an empty string. The two given ovjects are
     * converted to string and trimmed.
     */
    public static boolean stringEquals(Object obj1, Object obj2) {
        if (obj1 == obj2) return true;
        String str1 = (obj1 == null) ? "" : obj1.toString().trim();
        String str2 = (obj2 == null) ? "" : obj2.toString().trim();
        return str1.equals(str2);
    }


    /**
     * Compare the two objects (using equals(..)) taking care of null values.
     */
    public static boolean equals(Object obj1, Object obj2) {
        if (obj1 == obj2) return true;
        if ((obj1 == null) || (obj2 == null)) return false;
        return obj1.equals(obj2);
    }


    /**
     * Compare the two Comparable objects (using compareTo(..)) taking care of null values.
     */
    public static <T extends Comparable<T>> int compareTo(T obj1, T obj2) {
        if (obj1 == obj2) return 0;
        if (obj1 == null) return 1;
        if (obj2 == null) return -1;
        return obj1.compareTo(obj2);
    }


    /**
     * Compare the two Comparable objects (using compareTo(..)) taking care of null values. <br>
     * Note: this method uses a Collator instead of the simple <code>String.compareTo</code>.
     */
    public static int compareStrings(String str1, String str2) {
        if (str1 == str2) return 0;
        if (isEmpty(str1)) return isEmpty(str2) ? 0 : 1;
        if (isEmpty(str2)) return -1;
        return Collator.getInstance(Locale.FRENCH).compare(str1, str2);
    }


    /**
     * Compare the two enum objects (using ordinal()) taking care of null values.
     */
    public static <T extends Enum<T>> int compareTo(T obj1, T obj2) {
        if (obj1 == obj2) return 0;
        if (obj1 == null) return 1;
        if (obj2 == null) return -1;
        return (obj1.ordinal() < obj2.ordinal()) ? -1 : 1;
    }


    /**
     * A string is empty if it is null or contains only whitespace characters (space, tabs, returns, ...).
     */
    public static boolean isEmpty(String str) {
        if (str == null) return true;
        int len = str.length();
        for (int i = 0; i < len; i++) {
            if (str.charAt(i) > 32) return false;
        }
        return true;
    }


    /**
     * A string is empty if it is null or contains only whitespace characters (space, tabs, returns, ...).
     */
    public static boolean isEmpty(List<?> lst) {
        return (lst == null) || (lst.size() == 0);
    }


    /**
     * A string is defined if it is not null and don't contains only whitespace characters (space, tabs, returns, ...). <br>
     * <code>isEmpty(x) = !isDefined(x)</code>
     */
    public static boolean isDefined(String str) {
        if (str == null) return false;
        int len = str.length();
        for (int i = 0; i < len; i++) {
            if (str.charAt(i) > 32) return true;
        }
        return false;
    }


    /**
     * A list is defined if it is not null and it contains at least one element. <br>
     * <code>isEmpty(lst) = !isDefined(lst)</code>
     */
    public static boolean isDefined(List<?> lst) {
        return (lst != null) && (lst.size() > 0);
    }


    public static boolean startsWithIgnoreCase(String checkedString, String start) {
        if (isEmpty(start)) throw new NullPointerException("Matching start string cannot be null or empty");
        if (isEmpty(checkedString)) return false;
        int startLen = start.length();
        return checkedString.regionMatches(true, 0, start, 0, startLen);
    }

}
