/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.setting;


import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.swing.ComboBoxModel;
import javax.swing.ListModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import net.sf.sfac.utils.Comparison;


/**
 * Helper managing a list of sub-settings stored in a single setting. <br>
 * It allows to easily make profiles (= different set of identical settings saved in the same global settings file).
 * 
 * @author Olivier Berlanger
 */
public class SubSettingsList implements ListModel, ComboBoxModel {


    private Settings parent;
    private String keyPrefix;
    private List<SubSettingsProxy> subSettings;
    private List<ListDataListener> listeners;
    /** just for ComboBoxModel implementation */
    private Object selected;


    public SubSettingsList(Settings sett, String prefix) {
        parent = sett;
        keyPrefix = prefix;
        subSettings = new ArrayList<SubSettingsProxy>();
        initProxies();
    }


    /**
     * Synchronize the internal state of this object when parent settings is modified by outer code (for example: settings are
     * reloaded).
     */
    public void synchronizeWithSettings() {
        subSettings.clear();
        initProxies();
    }


    private void initProxies() {
        int[] indexes = parent.getIntArrayProperty(keyPrefix + ".indexes", new int[0]);
        int nbrSub = indexes.length;
        int index;
        for (int i = 0; i < nbrSub; i++) {
            index = indexes[i];
            subSettings.add(new SubSettingsProxy(parent, keyPrefix, index));
        }
        fireDataChanged();
    }


    public Settings getSubSettingAt(int i) {
        return subSettings.get(i);
    }


    public Settings addSubSetting() {
        return addSubSetting(subSettings.size(), null);
    }


    public Settings addSubSetting(int indexInList) {
        return addSubSetting(indexInList, null);
    }


    public Settings addSubSetting(int indexInList, String subSettingName) {
        int newIndex = getMaxIndex() + 1;
        SubSettingsProxy sett = new SubSettingsProxy(parent, keyPrefix, newIndex);
        subSettings.add(indexInList, sett);
        if (Comparison.isDefined(subSettingName)) sett.setName(subSettingName);
        sort();
        return sett;
    }


    public void removeSubSettings(Settings subSett) {
        SubSettingsProxy sett = (SubSettingsProxy) subSett;
        sett.clear();
        subSettings.remove(sett);
        if (selected == sett) selected = null;
        synchSettingIndexes();
        fireDataChanged();
    }


    public void removeSubSettingAt(int i) {
        SubSettingsProxy sett = subSettings.get(i);
        sett.clear();
        subSettings.remove(i);
        if (selected == sett) selected = null;
        synchSettingIndexes();
        fireDataChanged();
    }


    /**
     * Remove all the settings corresponding to this list from the parent settings.
     */
    public void clear() {
        for (Iterator<String> iter = parent.getKeys(); iter.hasNext();) {
            String key = iter.next();
            if (key.startsWith(keyPrefix)) {
                parent.removeProperty(key);
            }
        }
        subSettings.clear();
        synchSettingIndexes();
        fireDataChanged();
    }


    private int getMaxIndex() {
        int max = -1;
        for (SubSettingsProxy sett : subSettings) {
            if (max < sett.getIndex()) max = sett.getIndex();
        }
        return max;
    }


    private void synchSettingIndexes() {
        int nbrSett = subSettings.size();
        int[] indexes = new int[nbrSett];
        for (int i = 0; i < nbrSett; i++) {
            SubSettingsProxy sett = subSettings.get(i);
            indexes[i] = sett.getIndex();
        }
        parent.setIntArrayProperty(keyPrefix + ".indexes", indexes);
    }


    public void sort() {
        Collections.sort(subSettings);
        synchSettingIndexes();
        fireDataChanged();
    }


    // -------------------- ListModel & ComboBoxModel implementation ----------------------------------------


    public Object getElementAt(int index) {
        return subSettings.get(index);
    }


    public int getSize() {
        return subSettings.size();
    }


    public Object getSelectedItem() {
        return selected;
    }


    public void setSelectedItem(Object anItem) {
        selected = anItem;
    }


    public void addListDataListener(ListDataListener l) {
        if (listeners == null) listeners = new ArrayList<ListDataListener>();
        listeners.add(l);
    }


    public void removeListDataListener(ListDataListener l) {
        if (listeners != null) listeners.remove(l);
    }


    public void fireDataChanged() {
        if (listeners != null) {
            ListDataEvent ev = new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, 0, subSettings.size());
            for (ListDataListener l : listeners) {
                l.contentsChanged(ev);
            }
        }
    }


    public void fireSelectionChanged() {
        if (listeners != null) {
            ListDataEvent ev = new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, -1, -1);
            for (ListDataListener l : listeners) {
                l.contentsChanged(ev);
            }
        }
    }

}
