/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.lang;


import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import net.sf.sfac.utils.Comparison;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


public class LanguageSupportImpl extends LanguageSupport implements Cloneable {


    private static Log log = LogFactory.getLog(LanguageSupportImpl.class);

    /** The current locale used by this LanguageSupportImpl */
    private Locale currentLocale;
    /** All listeners registered for locale change events */
    private List<LanguageListener> listeners;
    /** Names of all the bundles used by this LanguageSupportImpl */
    private String[] bundleNames;
    /** All the current bundle instances of this LanguageSupportImpl (in the current locale) */
    private ResourceBundle[] bundles;
    /** Utility class tracking usage of strings (not null only in debug) */
    private LocalizedStringTracker tracker;


    /**
     * Construct a LanguageSupportImpl using one bundle and the given locale. Other bundles can be added with the
     * <code>addResourceBundle</code> method.
     * 
     * @param bundleName
     *            name of the (unique) bundle.
     * @param startLocaleId
     *            id of the locale.
     */
    public LanguageSupportImpl(String bundleName, String startLocaleId) {
        this((bundleName == null) ? null : new String[] { bundleName }, startLocaleId);
    }


    /**
     * Construct a LanguageSupportImpl using several bundle and the given locale.
     * 
     * @param allBundleNames
     *            name of the bundles.
     * @param startLocaleId
     *            id of the locale.
     */
    public LanguageSupportImpl(String[] allBundleNames, String startLocaleId) {
        if ((allBundleNames == null) || (allBundleNames.length == 0))
            throw new IllegalArgumentException("You must at least define one bundle");
        if (startLocaleId == null) throw new IllegalArgumentException("The start locale ID cannot be null");
        bundleNames = allBundleNames;
        bundles = new ResourceBundle[0];
        log.info("Create " + this + " for bundles: " + Arrays.toString(allBundleNames));
        updateLanguageImpl(startLocaleId);
    }


    @Override
    public LanguageSupport copyInstance() {
        try {
            LanguageSupportImpl cloned = (LanguageSupportImpl) super.clone();
            cloned.listeners = null;
            cloned.bundles = new ResourceBundle[0];
            cloned.currentLocale = null;
            // cloned.tracker = null;
            return cloned;
        } catch (CloneNotSupportedException cnse) {
            throw new InternalError("Sould never happen");
        }
    }


    /**
     * Set a tracker to track string usage in the application.
     */
    public void setTracker(LocalizedStringTracker newTracker) {
        tracker = newTracker;
        if (tracker != null) {
            for (int i = 0; i < bundles.length; i++) {
                for (Enumeration<String> keys = bundles[i].getKeys(); keys.hasMoreElements();) {
                    String key = keys.nextElement();
                    tracker.stringDefined(bundleNames[i], key, bundles[i].getString(key));
                }
            }
            tracker.init(bundleNames);
        }
    }


    public void saveTrackedInfo() {
        if (tracker != null) {
            try {
                tracker.saveAll();
            } catch (Exception e) {
                log.error("Unable to save LocalizedStringTracker info", e);
            }
        }
    }


    /**
     * Add a resourceBundle to the bundle used by this LanguageSupportImpl. (this will fire a language changed event to all
     * listeners).
     * 
     * @param bundleName
     *            newly defined bundle.
     */
    public void addResourceBundle(String bundleName) {
        if (log.isDebugEnabled()) log.debug("Add resource bundle: '" + bundleName + "'");
        int len = bundleNames.length;
        String[] newNames = new String[len + 1];
        System.arraycopy(bundleNames, 0, newNames, 0, len);
        newNames[len] = bundleName;
        bundleNames = newNames;
        updateLanguage(currentLocale.toString());
    }


    @Override
    public String getLocalizedStringImpl(String key) {
        return getLocalizedStringInternal(key, key, null);
    }


    @Override
    public String getLocalizedStringImpl(String key, Object... params) {
        return getLocalizedStringInternal(key, key, params);
    }


    @Override
    public String getOptionalLocalizedStringImpl(String key, String defaultValue) {
        return getLocalizedStringInternal(key, defaultValue, null);
    }


    private String getLocalizedStringInternal(String key, String defaultValue, Object[] params) {
        if (Comparison.isEmpty(key)) return "";
        String message = null;
        for (int i = bundles.length - 1; i >= 0; i--) {
            try {
                message = bundles[i].getString(key);
                if (tracker != null) tracker.stringUsed(bundleNames[i], key, message, (params == null) ? 0 : params.length);
                break;
            } catch (MissingResourceException mre) {
                // ignored -> search next bundle and issue a warning if nothing is found
            } catch (Exception e) {
                log.error("Internal string localization exception: ", e);
            }
        }
        if (message == null) {
            message = defaultValue;
            if (tracker != null) tracker.stringNotFound(key, defaultValue, (params == null) ? 0 : params.length);
            log.warn("Language item not found: " + key);
        }
        return (params == null) ? message : format(message, params);
    }


    private String format(String message, Object[] params) {
        int nbrObject = params.length;
        Object[] translatedParams = new Object[nbrObject];
        for (int i = 0; i < nbrObject; i++) {
            if (params[i] == null) {
                translatedParams[i] = "";
            } else if (params[i] instanceof MultiLingualText) {
                translatedParams[i] = ((MultiLingualText) params[i]).getText(currentLocale);
            } else {
                translatedParams[i] = params[i];
            }
        }
        return MessageFormat.format(message, translatedParams);
    }


    @Override
    public void updateLanguageImpl(String newLocaleId) {
        if (log.isDebugEnabled())
            log.debug("Call of updateLanguage for new locale: " + newLocaleId + " (currentLocale = " + currentLocale + ")");
        // check if new bundle was added.
        if (bundleNames.length != bundles.length) {
            if (log.isDebugEnabled()) log.debug("Adapt bundle array length " + bundles.length + " -> " + bundleNames.length);
            bundles = new ResourceBundle[bundleNames.length];
            // reset locale to force update
            currentLocale = null;
        }
        // update the bundles
        try {
            Locale newLocale = (newLocaleId == null) ? Locale.getDefault() : getLocale(newLocaleId);
            if ((currentLocale == null) || !currentLocale.equals(newLocale)) {
                log.info("Change application locale: " + currentLocale + " --> " + newLocale);
                try {
                    Locale.setDefault(newLocale);
                } catch (Exception e) {
                    log.warn("Failed to update default locale to: " + newLocale, e);
                }
                loadResources(newLocale);
                currentLocale = newLocale;
                fireLanguageChanged();
            }
        } catch (Exception e) {
            log.error("Locale resource loading error for locale: '" + newLocaleId + "'", e);
        }
    }


    protected void loadResources(Locale newLocale) {
        int len = bundleNames.length;
        for (int i = 0; i < len; i++) {
            if (log.isDebugEnabled())
                log.debug("  * Load bundle " + bundleNames[i] + "_" + newLocale + " (classloader = "
                        + getClass().getClassLoader() + ")");
            bundles[i] = ResourceBundle.getBundle(bundleNames[i], newLocale, getClass().getClassLoader());
        }
    }


    @Override
    public Locale getCurrentLocaleImpl() {
        return currentLocale;
    }


    /**
     * Get the locale object for the given locale ID.
     * 
     * @param localeId
     *            the locale ID.
     * @return The locale object for the given locale ID.
     */
    private static Locale getLocale(String localeId) {
        String language = "";
        String country = "";
        String variant = "";
        int firstUnderscore = localeId.indexOf('_');
        if (firstUnderscore < 0) {
            language = localeId;
        } else {
            language = localeId.substring(0, firstUnderscore);
            int secondUnderscore = localeId.indexOf('_', firstUnderscore + 1);
            if (secondUnderscore < 0) {
                country = localeId.substring(firstUnderscore + 1);
            } else {
                country = localeId.substring(firstUnderscore + 1, secondUnderscore);
                variant = localeId.substring(secondUnderscore + 1);
            }
        }
        return new Locale(language, country, variant);
    }


    // ----------------------- Language listener provider implementation ----------------------------------------------


    @Override
    public void addLanguageListenerImpl(LanguageListener lis) {
        if (listeners == null) listeners = new ArrayList<LanguageListener>();
        listeners.add(lis);
    }


    @Override
    public void removeLanguageListenerImpl(LanguageListener lis) {
        if (listeners != null) listeners.remove(lis);
    }


    private void fireLanguageChanged() {
        int len = (listeners == null) ? 0 : listeners.size();
        if (log.isDebugEnabled()) log.debug("Fire Locale changed to '" + currentLocale + "' to " + len + " listeners");
        for (int i = 0; i < len; i++) {
            LanguageListener lis = listeners.get(i);
            lis.languageChanged(currentLocale);
        }
    }


}
