/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.lang;


import java.util.Locale;


/**
 * Multilingual support: Class for the class managing the application locale and bundles. <br>
 * This class defines a singleton 'language service' allowing to get string values in the current locale and to register for
 * notification when locale changes. This 'language service' can find translations in a set of bundles.
 * 
 * @author Olivier Berlanger
 */
public abstract class LanguageSupport {


    private static LanguageSupport instance;


    /**
     * Get the unique LanguageSupport instance.
     * 
     * Note: this method is not synchronized because, as in all swing application, it is supposed to be accessed from a
     * single-threaded environment.
     * 
     * @return the global LanguageSupport instance (never null)
     */
    public static LanguageSupport getInstance() {
        if (instance == null) instance = new LanguageSupportMock();
        return instance;
    }


    /**
     * Set the unique LanguageSupport instance, any previous instance is discarded.
     * 
     * Note: this method is not synchronized because, as in all swing application, it is supposed to be accessed from a
     * single-threaded environment.
     */
    public static void setInstance(LanguageSupport newInstance) {
        instance = newInstance;
    }


    /**
     * Get the language-dependent string associated to the given key in the current locale. The key should be defined in the
     * LanguageSupport, if not, the string returned is the key itself and an warning message is logged.
     * 
     * @param key
     *            Key of the string in the LanguageSupport.
     * @return The language-dependent string associated to the given key in the current locale or the key itself if no value is
     *         found for the key.
     */
    public static String getLocalizedString(String key) {
        return getInstance().getLocalizedStringImpl(key);
    }


    /**
     * Get the language-dependent string associated to the given key in the current locale. The key should be defined in the
     * LanguageSupport, if not, the string returned is the key itself and an warning message is logged.
     * 
     * @param key
     *            Key of the string in the LanguageSupport.
     * @param params
     *            Parameters to place in the string value (using a TextFormat).
     * @return The language-dependent string associated to the given key in the current locale or the key itself if no value is
     *         found for the key.
     */
    public static String getLocalizedString(String key, Object... params) {
        return getInstance().getLocalizedStringImpl(key, params);
    }


    /**
     * Get an optional language-dependent string associated to the given key in the current locale. If the key is not defined in
     * the LanguageSupport, the default value is returned.
     * 
     * @param key
     *            Key of the string in the LanguageSupport.
     * @param defaultValue
     *            Default value to be returned if the string is not found in the LanguageSupport.
     * @return The language-dependent string associated to the given key in the current locale or the default value if no value
     *         is found for the key.
     */
    public static String getOptionalLocalizedString(String key, String defaultValue) {
        return getInstance().getOptionalLocalizedStringImpl(key, defaultValue);
    }


    /**
     * Ask the language manager to change current locale to the given one. If the given locale id is not the current one,
     * Resources (bundles) of the LanguageSupport will be updated and then an event will be fired to all listener.
     * 
     * @param newLocaleId
     *            ID of the new locale.
     */
    public static void updateLanguage(String newLocaleId) {
        getInstance().updateLanguageImpl(newLocaleId);
    }


    /**
     * Get the current Locale of the LanguageSupport.
     * 
     * @return The current Locale of the LanguageSupport.
     */
    public static Locale getCurrentLocale() {
        return getInstance().getCurrentLocaleImpl();
    }


    /**
     * Add a listener to be notified for all locale changes. <br>
     * Only a soft reference to the listeners will be kept by this class, so that it will not prevent garbage collection when
     * needed.
     * 
     * @param lis
     *            the new listener.
     */
    public static void addLanguageListener(LanguageListener lis) {
        getInstance().addLanguageListenerImpl(lis);
    }


    /**
     * Remove a previously added LanguageListener. (does nothing if the given listener was not added)
     * 
     * @param lis
     *            the listener to remove.
     */
    public static void removeLanguageListener(LanguageListener lis) {
        getInstance().removeLanguageListenerImpl(lis);
    }


    // ------------------ Abstract Implementation methods ------------------------------------


    public abstract LanguageSupport copyInstance();


    public abstract String getLocalizedStringImpl(String key);


    public abstract String getLocalizedStringImpl(String key, Object... params);


    public abstract String getOptionalLocalizedStringImpl(String key, String defaultValue);


    public abstract void updateLanguageImpl(String newLocaleId);


    public abstract Locale getCurrentLocaleImpl();


    public abstract void addLanguageListenerImpl(LanguageListener lis);


    public abstract void removeLanguageListenerImpl(LanguageListener lis);

}
