/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.file;


import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;

import javax.swing.JFileChooser;

import net.sf.sfac.lang.LanguageSupport;


/**
 * All knows file types descriptors/filters. Those <code>FileType</code> are mostly language independent, only the filter for
 * the JFileChooser (inner class FileChooserFilter) contains a language-dependent description.
 * 
 * @author Olivier Berlanger
 */
public enum FileType implements FileFilter, FilenameFilter {

    FILE_TYPE_ALL("All files (*.*)", null), 
    FILE_TYPE_TEXT("Text files (*.txt)", new String[] { "txt", "text" }), 
    FILE_TYPE_XML("XML files (*.xml)", new String[] { "xml" }), 
    FILE_TYPE_PROPERTIES("Properties files (*.properties)", new String[] { "properties" }), 
    FILE_TYPE_PNG("Portable Network Graphics (*.png)", new String[] { "png" }), 
    FILE_TYPE_JPEG("Jpeg images (*.jpg, *.jpeg)", new String[] { "jpg", "jpeg" }), 
    FILE_TYPE_GIF("Gif images (*.gif)", new String[] { "gif" }), 
    FILE_TYPE_IMAGES("All images (*.jpg, *.gif, *.png)", new String[] { "jpg", "png", "gif", "jpeg" }), 
    FILE_TYPE_PDF("Portable Document Format (*.pdf)", new String[] { "pdf" }), 
    FILE_TYPE_ZIP("Compressed archives (*.zip)", new String[] { "zip" }), 
    FILE_TYPE_CBZ("Comic Book (*.cbz)", new String[] { "cbz" }), 
    FILE_TYPE_HTML("HTML files (*.html, *.htm)", new String[] { "html", "htm" }), 
    FILE_TYPE_DB("Database files (*.db4o)", new String[] { "db4o" }), 
    FILE_TYPE_CSV("Coma separated (*.csv)", new String[] { "csv" }), 
    FILE_TYPE_FO("XSL-FO xml documents (*.fo)", new String[] { "fo", "xsl-fo" }), 
    FILE_TYPE_SONG("Song (*.sng)", new String[] { "sng" }), 
    FILE_TYPE_MP3("Audio (*.mp3)", new String[] { "mp3" }), 
    FILE_TYPE_VIDEO("Movie (*.mp4, *.avi, *.mov)", new String[] { "mp4", "avi", "mov" });


    /**
     * Get the type of the given file. The returned value will be one of the given possible type or the given defaultType. The
     * type FILE_TYPE_ALL is ignored if it's present in the possible types.
     * 
     * @param fil
     *            the file of wich we want to know the type.
     * @param possibleTypes
     *            all the possible types.
     * @param defaultType
     *            the type returned if none of the given types matched.
     * @return int the file type.
     */
    public static FileType getFileType(File fil, FileType[] possibleTypes, FileType defaultType) {
        int len = possibleTypes.length;
        FileType type;
        for (int i = 0; i < len; i++) {
            type = possibleTypes[i];
            if ((type != FILE_TYPE_ALL) && type.isFileOfType(fil)) return type;
        }
        return defaultType;
    }


    /**
     * Add the correct extension to the given file.
     * 
     * @param fil
     *            The file to be updated with correct extension.
     * @param type
     *            The type of the file (defining the extension it should have)
     * @param overrideExisting
     *            if true any existing extension not matching the expected one will be replaced. If false, the correct extension
     *            is only added if there is no extension currently defined.
     * @return File the given file with possibly updated extension.
     */
    public static File setFileExtension(File fil, FileType type, boolean overrideExisting) {
        if (type.isFileOfType(fil)) return fil;
        String defaultExt = type.getDefaultExtension();
        if (defaultExt == null) return fil;
        String fileName = fil.getName();
        String newFileName;
        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex < 0) {
            newFileName = fileName + '.' + defaultExt;
        } else {
            if (overrideExisting) {
                newFileName = fileName.substring(0, lastDotIndex + 1) + defaultExt;
            } else {
                return fil;
            }
        }
        File parentDir = fil.getParentFile();
        File newFile = (parentDir == null) ? new File(newFileName) : new File(parentDir, newFileName);
        return newFile;
    }


    /**
     * Add choosable file filter(s) to the given file chooser.
     * 
     * @param chooser
     *            the file chooser.
     * @param types
     *            the file types for wich we want choosable file filter(s). (you can pass null if you want only set the 'select
     *            all (*.*)' filter)
     * @param addAllFilter
     *            true if the custom (= language aware) 'select all (*.*)' filter should be added.
     */
    public static void addChoosableFileFilters(JFileChooser chooser, FileType[] types, boolean addAllFilter) {
        if (addAllFilter) {
            chooser.setAcceptAllFileFilterUsed(false);
            chooser.addChoosableFileFilter(FILE_TYPE_ALL.getFileChooserFilter());
        }
        int len = (types == null) ? 0 : types.length;
        // reverse iterate so the first type (wich is usually the default one) will be selected.
        for (int i = len - 1; i >= 0; i++) {
            chooser.addChoosableFileFilter(types[i].getFileChooserFilter());
        }
    }


    /**
     * Utility method to get the extension of a file name (without the dot).
     * 
     * @param fileName
     *            The full file name (like: "myFile.txt").
     * @return String The file extension (like "txt").
     */
    public static String getFileExtension(String fileName) {
        String ext = null;
        int i = fileName.lastIndexOf('.');
        if ((i > 0) && (i < fileName.length() - 1)) ext = fileName.substring(i + 1).toLowerCase();
        return ext;
    }


    // ---------------------------- instance implementation --------------------------


    private String defaultDescription;
    private String[] extensions;


    /**
     * Create a file type descriptors/filters.
     * 
     * @param typeTag
     *            Tag of the file type (for clarity it should start with "FILE_TYPE_")
     * @param typeDescription
     *            Default english description of the file.
     * @param fileExtensions
     *            The accepted file extensions.
     */
    private FileType(String typeDescription, String[] fileExtensions) {
        defaultDescription = typeDescription;
        extensions = fileExtensions;
    }


    public String getDefaultDescription() {
        return defaultDescription;
    }


    public String getDefaultExtension() {
        return (extensions == null) ? null : extensions[0];
    }


    public String[] getAcceptedExtensions() {
        return extensions;
    }


    /**
     * Return true if the given file has the good extension or if the given file is a directory. <br>
     * Returns false otherwise.
     */
    public boolean accept(File f) {
        if (f.isDirectory()) return true;
        String extension = getFileExtension(f.getName());
        return isAcceptedExtension(extension);
    }


    /**
     * Return true if the given file name has the good extension. <br>
     * Note: the given directory is ignored.
     */
    public boolean accept(File dir, String name) {
        String extension = getFileExtension(name);
        return isAcceptedExtension(extension);
    }


    /**
     * Check if the given file is of the specifed type. (the type check is only done on the file name extension). <br>
     * This method always returns false for directories. If the given file is null, false is returned.
     */
    public boolean isFileOfType(File fil) {
        if ((fil == null) || fil.isDirectory()) return false;
        return accept(fil);
    }


    public boolean isAcceptedExtension(String extension) {
        if (extensions == null) return true;
        int len = extensions.length;
        for (int i = 0; i < len; i++)
            if (extensions[i].equalsIgnoreCase(extension)) return true;
        return false;
    }


    public String toString() {
        return getClass().getSimpleName() + '[' + name() + ']';
    }


    /**
     * Create a <code>JFileChooser</code> choosable file filter for this file type.
     * 
     * @return FileChooserFilter a new choosable file filter for this file type.
     */
    FileChooserFilter getFileChooserFilter() {
        String descr = defaultDescription;
        String key = name() + "_DESCR";
        descr = LanguageSupport.getLocalizedString(key);
        if (descr == key) descr = defaultDescription;
        return new FileChooserFilter(descr);
    }


    /**
     * Chooseable file filter for <code>JFileChooser</code>. This class contains a language-dependent description. As it doesn't
     * listen to the LanguageManagger, it must be re-created for each use.
     * 
     * @author Olivier Berlanger
     */
    class FileChooserFilter extends javax.swing.filechooser.FileFilter {


        String chooserDescription;


        FileChooserFilter(String descr) {
            chooserDescription = descr;
        }


        public boolean accept(File f) {
            return FileType.this.accept(f);
        }


        /**
         * The description of this filter. For example: "JPG and GIF Images"
         */
        public String getDescription() {
            return chooserDescription;
        }

    }

}
